<?php

namespace Drupal\acquia_optimize;

use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;

/**
 * API client service.
 */
class ApiClient {

  use StringTranslationTrait;

  /**
   * API key.
   */
  private string $apiKey;

  /**
   * API URL.
   */
  private string $apiUrl;

  /**
   * Accessibility level.
   */
  private string $accessibilityLevel;

  /**
   * Constructor.
   *
   * @param \GuzzleHttp\ClientInterface $client
   *   The Guzzle client.
   * @param \Drupal\Core\Asset\LibraryDiscoveryInterface $libraryDiscovery
   *   The library discovery service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    private readonly ClientInterface $client,
    private readonly LibraryDiscoveryInterface $libraryDiscovery,
    ConfigFactoryInterface $configFactory,
  ) {
    $config = $configFactory->get('acquia_optimize.settings');
    $this->apiKey = $config->get('api_key') ?? "";
    $this->apiUrl = $config->get('api_url') ?? "";
    $this->accessibilityLevel = $config->get('accessibility') ?? "";
  }

  /**
   * Test API connection.
   *
   * @return array
   *   Response message.
   */
  public function testApi(): array {
    return $this->request('/account', 'GET');
  }

  /**
   * Create scan request.
   *
   * @param string $encodedPage
   *   The encoded page.
   * @param string $html
   *   The HTML to scan.
   * @param string $css
   *   The CSS to scan.
   *
   * @return array
   *   The scan request response.
   */
  public function createScanRequest(string $encodedPage, string $html, string $css): array {
    return $this->request('/html_scans', 'POST', [
      'encoded_page' => $encodedPage,
      'html' => $html,
      'css' => $css,
      'accessibility' => $this->accessibilityLevel,
    ]);
  }

  /**
   * Check scan status.
   *
   * @param string $scanId
   *   The scan ID.
   *
   * @return array
   *   The scan status response.
   */
  public function checkScanStatus(string $scanId): array {
    return $this->request('/html_scans/' . $scanId, 'GET');
  }

  /**
   * Request.
   *
   * @param string $endpoint
   *   Endpoint to call for the Acquia Optimize API.
   * @param string $method
   *   HTTP method to use.
   * @param array $body
   *   Request body defaults to empty array.
   *
   * @return array
   *   Response or error object.
   */
  private function request(string $endpoint, string $method, array $body = []): array {
    $path = $this->libraryDiscovery->getLibraryByName('acquia_optimize', 'optimize')['js'][0]['data'];
    if (!file_exists($path)) {
      return [
        'status' => 404,
        'error' => $this->t('The Acquia Optimize library is not installed. Please obtain the library from Acquia and place it at <strong>@path</strong>.', ['@path' => $path]),
      ];
    }

    if (empty($this->apiKey)) {
      return [
        'status' => 400,
        'error' => $this->t('Missing API key. Please obtain an Acquia Optimize API key and enter it in the settings page.'),
      ];
    }

    $options = [
      'headers' => [
        'Authorization' => 'Bearer ' . $this->apiKey,
      ],
    ];
    if (!empty($body)) {
      $options['json'] = $body;
    }

    try {
      $response = $this->client->request($method, $this->apiUrl . $endpoint, $options);
      $responseBody = $response->getBody()->getContents();
      $responseArray = json_decode($responseBody, TRUE, 512, JSON_THROW_ON_ERROR);
      return $responseArray;
    }
    catch (\Exception $e) {
      return ['error' => $e->getMessage(), 'status' => $e->getCode() ?? 500];
    }
  }

}
