<?php

namespace Drupal\acquia_optimize;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Contains utility functions for module.
 */
class Utilities {

  /**
   * Configuration object.
   *
   * @var \Drupal\Core\Config\Config
   */
  private $config;

  /**
   * Module Path.
   *
   * @var string
   */
  private $modulePath;

  /**
   * Constructor for the Utilities class.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(ConfigFactoryInterface $configFactory, ModuleHandlerInterface $moduleHandler) {
    $this->config = $configFactory->get('acquia_optimize.settings');
    $this->modulePath = $moduleHandler->getModule('acquia_optimize')->getPath();
  }

  /**
   * Accessibility guidelines.
   *
   * @var array
   */
  public const ACCESSIBILITY_GUIDELINES = [
    'WCAG2-A',
    'WCAG2-AA',
    'WCAG2-AAA',
    'WCAG21-A',
    'WCAG21-AA',
    'WCAG21-AAA',
    'WCAG22-A',
    'WCAG22-AA',
    'WCAG22-AAA',
  ];

  /**
   * Map of SEO issues to friendly names.
   *
   * @var array
   */
  private const SEO_MAPPING = [
    'seo_missing_title' => 'Missing title',
    'seo_duplicate_title' => 'Title found on more than one page',
    'seo_too_long_url' => 'Too long URL',
    'seo_missing_h1' => 'Missing H1',
    'seo_multiple_h1' => 'Multiple H1 on page',
    'seo_duplicate_h1' => 'H1 found on more than one page',
    'seo_missing_meta_description' => 'Missing META description',
    'seo_too_short_meta_description' => 'Too short META description',
    'seo_too_long_meta_description' => 'Too long META description',
    'seo_too_many_page_links' => 'Too many internal links',
    'seo_missing_subheadlines' => 'Missing sub headings',
    'seo_missing_alt' => 'Images missing ALT',
    'seo_no_index' => 'Pages with - No index',
    'seo_no_follow' => 'Links with - No follow',
  ];

  /**
   * Map of error messages.
   */
  private const ERRORS = [
    500 => 'Could not connect to the Acquia Optimize API due to a server error. Please try again later.',
    422 => 'Access to the Acquia Optimize API was denied. Contact your administrator for help.',
    408 => 'The scan request has timed out. Please try again later.',
    400 => 'There was an error gathering the content. Please try again.',
  ];

  /**
   * Get SEO mapping.
   */
  public function getSeoMapping(string $seoIssue): string {
    return self::SEO_MAPPING[$seoIssue] ?? ucwords(str_replace('_', ' ', $seoIssue));
  }

  /**
   * Get error message.
   */
  public function buildErrorMessage(int $statusCode): string {
    $image_url = $this->getImagePath('error.svg');

    if (str_starts_with((string) $statusCode, '5')) {
      $statusCode = 500;
    }
    $errorMessage = self::ERRORS[$statusCode] ?? 'An unknown error occurred';
    return '<div class="scan__has-failed"><img src="' . $image_url . '" /><p>' . $errorMessage . '</p></div>';
  }

  /**
   * Get image path.
   */
  public function getImagePath(string $imageName): string {
    return base_path() . $this->modulePath . '/images/' . $imageName;
  }

  /**
   * Reduce accessibility issues by friendly name.
   */
  public function getReducedAccessibilityIssues(array $accessibilityIssues): array {
    return array_reduce($accessibilityIssues, function ($acc, $obj) {
      if (!isset($acc[$obj['friendly_name']])) {
        $acc[$obj['friendly_name']] = [];
      }
      $acc[$obj['friendly_name']][] = $obj;
      return $acc;
    }, []);
  }

  /**
   * Group accessibility issues into friendly name and items.
   */
  public function getGroupedAccessibilityIssues(array $reducedAccessibilityIssues): array {
    return array_map(function ($key) use ($reducedAccessibilityIssues) {
      $sanitizedKey = Xss::filter($key);
      return ['friendly_name' => $sanitizedKey, 'items' => $reducedAccessibilityIssues[$key]];
    }, array_keys($reducedAccessibilityIssues));
  }

  /**
   * Group SEO issues into name and count.
   */
  public function getGroupedSeoIssues(array $seoIssues): array {
    return array_reduce($seoIssues, function ($acc, $issue) {
      $name = self::getSeoMapping($issue['name']);
      $classification = $issue['classification_name'];
      if (!isset($acc[$name])) {
        $acc[$name] = [
          'count' => 0,
          'classification' => $classification,
        ];
      }
      $acc[$name]['count']++;
      return $acc;
    }, []);
  }

  /**
   * Check if an API key has been set.
   */
  public function apiKeyHasBeenSet(): bool {
    return !empty($this->config->get('api_key'));
  }

}
