<?php

/**
 * @file
 * Drupal configuration file for sites on Acquia Cloud Site Factory.
 *
 * This file is included from the site specific settings.php file for any site
 * registered with Site Factory. If no site is found, sites/default/settings.php
 * is read instead of this file (in normal circumstances).
 */

use Drush\Drush;

// Acquia rules disallow 'include/require' with dynamic arguments. We extend the
// exception over all pasted Drupal settings.php comments so that we don't need
// to change the PHPDoc headers to end with a full stop, or change # to //.
// phpcs:disable

// Include custom settings.php code from factory-hooks/pre-settings-php.
if (function_exists('acsf_hooks_includes')) {
  foreach (acsf_hooks_includes('pre-settings-php') as $_acsf_include_file) {
    // This should not use include_once / require_once. Some Drush versions do
    // Drupal bootstrap multiple times, and include_once / require_once would
    // make the hook modifications not be included on the second bootstrap.
    include $_acsf_include_file;
  }
}

/**
 * PHP settings:
 *
 * To see what PHP settings are possible, including whether they can be set at
 * runtime (by using ini_set()), read the PHP documentation:
 * http://php.net/manual/ini.list.php
 * See \Drupal\Core\DrupalKernel::bootEnvironment() for required runtime
 * settings and the .htaccess file for non-runtime settings.
 * Settings defined there should not be duplicated here so as to avoid conflict
 * issues.
 */

/**
 * If you encounter a situation where users post a large amount of text, and
 * the result is stripped out upon viewing but can still be edited, Drupal's
 * output filter may not have sufficient memory to process it.  If you
 * experience this issue, you may wish to uncomment the following two lines
 * and increase the limits of these variables.  For more information, see
 * http://php.net/manual/pcre.configuration.php.
 */
# ini_set('pcre.backtrack_limit', 200000);
# ini_set('pcre.recursion_limit', 200000);

/**
 * Fast 404 pages:
 *
 * Drupal can generate fully themed 404 pages. However, some of these responses
 * are for images or other resource files that are not displayed to the user.
 * This can waste bandwidth, and also generate server load.
 *
 * The options below return a simple, fast 404 page for URLs matching a
 * specific pattern:
 * - $config['system.performance']['fast_404']['exclude_paths']: A regular
 *   expression to match paths to exclude, such as images generated by image
 *   styles, or dynamically-resized images. The default pattern provided below
 *   also excludes the private file system. If you need to add more paths, you
 *   can add '|path' to the expression.
 * - $config['system.performance']['fast_404']['paths']: A regular expression to
 *   match paths that should return a simple 404 page, rather than the fully
 *   themed 404 page. If you don't have any aliases ending in htm or html you
 *   can add '|s?html?' to the expression.
 * - $config['system.performance']['fast_404']['html']: The html to return for
 *   simple 404 pages.
 *
 * Acquia Cloud Site Factory compatibility notes:
 *
 * The Drupal 7 version of the ACSF module hardcodes these configuration values;
 * for Drupal 8 it is left up to the specific installation to override these,
 * e.g. in a post-settings-php hook, or by including these settings in .yml
 * configuration files according to the customer's own workflow.
 *
 * There is currently one difference of the default 'paths' value for Drupal 8
 * compared to the hardcoded value in the ACSF module for Drupal 7: svg files
 * are not included here.
 */
# $config['system.performance']['fast_404']['exclude_paths'] = '/\/(?:styles)|(?:system\/files)\//';
# $config['system.performance']['fast_404']['paths'] = '/\.(?:txt|png|gif|jpe?g|css|js|ico|swf|flv|cgi|bat|pl|dll|exe|asp)$/i';
# $config['system.performance']['fast_404']['html'] = '<!DOCTYPE html><html><head><title>404 Not Found</title></head><body><h1>Not Found</h1><p>The requested URL "@path" was not found on this server.</p></body></html>';

// phpcs:enable

/**
 * Acquia Cloud Site Factory specific settings.
 */
if (file_exists('/var/www/site-php')) {
  // This global variable is set during the 'configuration' (sites.php)
  // bootstrap phase. Notes:
  // - The 'env' value contains the 'canonical environment' which (unlike
  //   $_ENV['AH_SITE_ENVIRONMENT']) stays the same during code deployments
  //   where sites are moved between two environments.
  // - post-settings-php hooks should use $GLOBALS['gardens_site_settings']
  //   rather than the below 'local' variables which are in principle not
  //   guaranteed to stay defined. (We still define $env / $role for some
  //   existing customers' post-settings-php hooks that already use them.)
  $site_settings = !empty($GLOBALS['gardens_site_settings'])
    ? $GLOBALS['gardens_site_settings']
    : ['site' => '', 'env' => '', 'conf' => ['acsf_db_name' => '']];
  $env = $site_settings['env'];
  $role = $site_settings['conf']['acsf_db_name'];

  $drupal_version = 8;
  if (version_compare(\Drupal::VERSION, '9', '>=') && version_compare(\Drupal::VERSION, '10', '<')) {
    $drupal_version = 9;
  }
  elseif (version_compare(\Drupal::VERSION, '10', '>=') && version_compare(\Drupal::VERSION, '11', '<')) {
    $drupal_version = 10;
  }
  elseif (version_compare(\Drupal::VERSION, '11', '>=')) {
    $drupal_version = 11;
  }
  $_acsf_include_file = "/var/www/site-php/{$site_settings['site']}.{$site_settings['env']}/D{$drupal_version}-{$site_settings['env']}-{$site_settings['conf']['acsf_db_name']}-settings.inc";
  if (file_exists($_acsf_include_file)) {
    // Acquia rules disallow 'include/require' with dynamic arguments.
    // phpcs:disable
    include $_acsf_include_file;
    // phpcs:enable
    // Overwrite trusted_host_patterns setting, remove unnecessary hosts.
    // Allowed hosts for D8: https://www.drupal.org/node/2410395.
    // The overwrite doesn't cause any security problem because the valid hosts
    // were checked before in our sites.json registry.
    $settings['trusted_host_patterns'] = [
      '^' . str_replace('*', '.+',
        str_replace('.', '\.', $_SERVER['HTTP_HOST'])
      ) . '$',
    ];
  }
  elseif (PHP_SAPI === 'cli') {
    throw new Exception("No database connection file was found for DB {$site_settings['conf']['acsf_db_name']}.");
  }
  else {
    syslog(LOG_ERR, "GardensError: AN-22471 - No database connection file was found for DB {$site_settings['conf']['acsf_db_name']}.");
    header($_SERVER['SERVER_PROTOCOL'] . ' 503 Service unavailable');
    print 'The website encountered an unexpected error. Please try again later.';
    exit;
  }
  // todo: this part needs to be rewritten, we might consider removing it
  // entirely for the time being.
  if (!class_exists('DrupalFakeCache')) {
    $config['cache_backends'][] = 'includes/cache-install.inc';
  }
  // Rely on the external Varnish cache for page caching.
  $config['cache_class_cache_page'] = 'DrupalFakeCache';
  $config['cache'] = 1;
  $config['page_cache_maximum_age'] = 300;
  // We can't use an external cache if we are trying to invoke these hooks.
  $config['page_cache_invoke_hooks'] = FALSE;

  // This section has been ported from D7 to D8 by mistake; the 'memcache_*'
  // settings are not supposed to be set and the $config changes aren't generic.
  // todo: reevaluate and possibly remove this.
  if (!empty($site_settings['flags']['memcache_enabled']) && !empty($site_settings['memcache_inc'])) {
    $config['cache_backends'][] = $site_settings['memcache_inc'];
    $config['cache_default_class'] = 'MemCacheDrupal';
    $config['cache_class_cache_form'] = 'DrupalDatabaseCache';
    // The oembed cache in many cases should not evict data (given that data
    // is obtained from costly API calls and is not expected to change when
    // refreshed), so is more suited to the database than to memcache.
    $config['cache_class_cache_oembed'] = 'DrupalDatabaseCache';
  }

  // Until the site installation finishes, noone should be able to visit the
  // site, unless the site is being installed via install.php and the user has
  // the correct token to access it.
  if (PHP_SAPI !== 'cli' && !empty($site_settings['flags']['access_restricted']['enabled'])) {
    $_tmp = !empty($site_settings['flags']['access_restricted']['token']) && !empty($_GET['site_install_token']) && $_GET['site_install_token'] === $site_settings['flags']['access_restricted']['token'];
    if (!$_tmp || $_SERVER['SCRIPT_NAME'] !== $GLOBALS['base_path'] . 'install.php') {
      header($_SERVER['SERVER_PROTOCOL'] . ' 503 Service unavailable');
      if (!empty($site_settings['flags']['access_restricted']['reason'])) {
        print $site_settings['flags']['access_restricted']['reason'];
      }
      exit;
    }
  }
  if (isset($_ENV['AH_SITE_ENVIRONMENT'])) {
    // DG-10819: Enable Migrate background operations by default on all Acquia
    // hosting environments. See https://drupal.org/node/1958170. The path here
    // should be valid on all Acquia hosting servers, and will not take effect
    // on non-Acquia environments since AH_SITE_ENVIRONMENT won't be set in that
    // case.
    $config['migrate_drush_path'] = '/usr/local/bin/drush';
  }

  // Do not override the private path if the customer has defined its value
  // in a pre-settings-php hook.
  if (empty($settings['file_private_path']) && !empty($site_settings['file_private_path'])) {
    $settings['file_private_path'] = $site_settings['file_private_path'];
  }

  // Do not override the public path if the customer has defined its value
  // in a pre-settings-php hook.
  if (empty($settings['file_public_path']) && !empty($site_settings['file_public_path'])) {
    $settings['file_public_path'] = $site_settings['file_public_path'];
  }

  // Propagate all configuration values in ACSF per-site storage into Drupal's
  // configuration.
  if (!empty($site_settings['conf']) && is_array($site_settings['conf'])) {
    $config = $site_settings['conf'] + $config;
  }
}

/**
 * Location of the site configuration files.
 *
 * The $config_directories array specifies the location of file system
 * directories used for configuration data. On install, "active" and "sync"
 * directories are created for configuration. The sync directory is used for
 * configuration imports; the active directory is not used by default, since the
 * default storage for active configuration is the database rather than the file
 * system (this can be changed; see "Active configuration settings" below).
 *
 * The default location for the active and sync directories is inside a
 * randomly-named directory in the public files path; this setting allows you to
 * override these locations. If you use files for the active configuration, you
 * can enhance security by putting the active configuration outside your
 * document root.
 *
 * Example:
 * @code
 *   $config_directories = array(
 *     CONFIG_SYNC_DIRECTORY => '/another/directory/outside/webroot',
 *   );
 * @endcode
 */
if (isset($config_directories['vcs'])) {
  // The hosting settings include file adds a VCS config directory, but this can
  // only work with livedeev enabled.  Livedev is not supported on ACSF, and the
  // addition of this directory breaks site installs, so the VCS config
  // directory is removed for now.
  // @see https://backlog.acquia.com/browse/CL-11815
  // @see https://github.com/drush-ops/drush/pull/1711
  if (class_exists('\Drush\Drush') && Drush::hasContainer()) {
    try {
      $_tmp = Drush::input()->getArgument('command');
    }
    catch (InvalidArgumentException $e) {
      $_tmp = FALSE;
    }

    if ($_tmp === 'site-install') {
      unset($config_directories['vcs']);
    }
  }
}

// Include custom settings.php code from factory-hooks/post-settings-php.
if (function_exists('acsf_hooks_includes')) {
  foreach (acsf_hooks_includes('post-settings-php') as $_acsf_include_file) {
    // Acquia rules disallow 'include/require' with dynamic arguments.
    // phpcs:disable
    include $_acsf_include_file;
    // phpcs:enable
  }
}
