<?php

namespace Drupal\Tests\ad_content\Functional;

use Drupal\ad_content\Entity\AdContent;
use Drupal\ad_content\Entity\AdContentType;
use Drupal\ad\Entity\AdPlacement;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the AdContent entity CRUD operations.
 *
 * @group ad_content
 */
class AdContentTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['ad', 'ad_content', 'ad_track', 'block'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to administer ads.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A test ad content type.
   *
   * @var \Drupal\ad_content\Entity\AdContentType
   */
  protected $adContentType;

  /**
   * A test ad placement.
   *
   * @var \Drupal\ad\Entity\AdPlacement
   */
  protected $placement;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer ads',
      'administer ad settings',
      'view ads',
    ]);

    // Create a test placement.
    $this->placement = AdPlacement::create([
      'id' => 'test_placement',
      'label' => 'Test Placement',
      'description' => 'Test description',
    ]);
    $this->placement->save();

    // Create a test ad content type.
    $this->adContentType = AdContentType::create([
      'id' => 'test_type',
      'label' => 'Test Type',
    ]);
    $this->adContentType->save();
  }

  /**
   * Tests creating an ad content entity.
   */
  public function testCreateAdContent() {
    $this->drupalLogin($this->adminUser);

    // Navigate to the add form.
    $this->drupalGet('/admin/content/ad/add/test_type');
    $this->assertSession()->statusCodeEquals(200);

    // Fill in the form.
    $edit = [
      'title[0][value]' => 'Test Advertisement',
      'placement' => 'test_placement',
      'target_url[0][uri]' => 'https://example.com',
      'status[value]' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    // Verify the ad was created.
    $this->assertSession()->pageTextContains('Test Advertisement');

    // Verify we can load the ad.
    $ads = \Drupal::entityTypeManager()
      ->getStorage('ad_content')
      ->loadByProperties(['title' => 'Test Advertisement']);
    $this->assertNotEmpty($ads);
    $ad = reset($ads);
    $this->assertEquals('Test Advertisement', $ad->label());
    $this->assertEquals('test_placement', $ad->get('placement')->value);
  }

  /**
   * Tests deleting an ad content entity.
   */
  public function testDeleteAdContent() {
    $this->drupalLogin($this->adminUser);

    // Create an ad first.
    $ad = AdContent::create([
      'bundle' => 'test_type',
      'title' => 'Test Ad',
      'placement' => 'test_placement',
      'target_url' => ['uri' => 'https://example.com'],
      'status' => TRUE,
    ]);
    $ad->save();
    $adId = $ad->id();

    // Delete the ad.
    $this->drupalGet('/admin/content/ad/' . $adId . '/delete');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete');

    $this->submitForm([], 'Delete');

    // Verify the ad was deleted.
    $ad = AdContent::load($adId);
    $this->assertNull($ad);
  }

  /**
   * Tests that placement field is required.
   */
  public function testPlacementFieldRequired() {
    $this->drupalLogin($this->adminUser);

    // Try to create an ad without a placement.
    $this->drupalGet('/admin/content/ad/add/test_type');
    $edit = [
      'title[0][value]' => 'Test Ad',
      'target_url[0][uri]' => 'https://example.com',
      'status[value]' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    // Should show validation error for required placement field.
    $this->assertSession()->pageTextContains('Placement field is required');
  }

  /**
   * Tests that an advertisement is displayed in an advertisement block.
   */
  public function testAdDisplayedInBlock() {
    // Create an ad content entity.
    $ad = AdContent::create([
      'bundle' => 'test_type',
      'title' => 'Front Page Advertisement',
      'placement' => 'test_placement',
      'target_url' => ['uri' => 'https://example.com'],
      'status' => TRUE,
    ]);
    $ad->save();

    // Place an ad block on the front page.
    $this->drupalPlaceBlock('ad', [
      'id' => 'test_ad_block',
      'region' => 'content',
      'label' => 'Test Ad Block',
      'bucket_ids' => ['ad_content'],
      'placement_id' => 'test_placement',
    ]);

    // Visit the front page as a user with view ads permission.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // Verify the ad placeholder element is present in the page.
    // The block renders a placeholder that gets filled via AJAX.
    $this->assertSession()->pageTextContains('Test Ad Block');
  }

}
