<?php

namespace Drupal\add_content_by_bundle\Plugin\views\area;

use Drupal\Component\Utility\Html;
use Drupal\Core\Access\AccessManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RedirectDestinationTrait;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an area plugin to display a bundle-specific node/add link.
 *
 * @ingroup views_area_handlers
 *
 * @ViewsArea("add_content_by_bundle")
 */
class AddContentByBundle extends AreaPluginBase {
  use RedirectDestinationTrait;

  /**
   * The access manager.
   *
   * @var \Drupal\Core\Access\AccessManagerInterface
   */
  protected $accessManager;

  /**
   * Constructs a new plugin instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Access\AccessManagerInterface $access_manager
   *   The access manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AccessManagerInterface $access_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->accessManager = $access_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('access_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['bundle'] = ['default' => NULL];
    $options['label'] = ['default' => NULL];
    $options['class'] = ['default' => NULL];
    $options['target'] = ['default' => ''];
    $options['width'] = ['default' => '600'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $entity_type = $this->view->getBaseEntityType()->id();
    // All bundle types.
    $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo($entity_type);
    // Options list.
    $bundlesList = [];
    foreach ($bundles as $id => $bundle) {
      $label = $bundle['label'];
      $bundlesList[$id] = $label;
    }
    // New content bundle type.
    // TODO: preselect if a single bundle specified for the view?
    $form['bundle'] = [
      '#title' => $this->t('Add content bundle (Content) type'),
      '#description' => $this->t('The bundle (content) type of content to add.'),
      '#type' => 'select',
      '#options' => $bundlesList,
      '#default_value' => (!empty($this->options['bundle'])) ? $this->options['bundle'] : '',
      '#required' => TRUE,
    ];
    $form['label'] = [
      '#title' => $this->t('Label'),
      '#description' => $this->t('The text of the link.'),
      '#type' => 'textfield',
      '#default_value' => $this->options['label'] ?: $this->t('Add a new entry'),
      '#required' => TRUE,
    ];
    // TODO: allow for multiple classes.
    $form['class'] = [
      '#title' => $this->t('Class'),
      '#description' => $this->t('A CSS class to apply to the link. If using multiple classes, separate them by spaces.'),
      '#type' => 'textfield',
      '#default_value' => $this->options['class'],
    ];
    $form['target'] = [
      '#title' => $this->t('Target'),
      '#description' => $this->t('Optionally have the form open on-page in a modal or off-canvas dialog.'),
      '#type' => 'select',
      '#default_value' => $this->options['target'],
      '#options' => [
        '' => $this->t('Default'),
        'tray' => $this->t('Off-Screen Tray'),
        'modal' => $this->t('Modal Dialog'),
      ],
    ];
    $form['width'] = [
      '#title' => $this->t('Dialog Width'),
      '#description' => $this->t('How wide the dialog should appear.'),
      '#type' => 'number',
      '#min' => '100',
      '#default_value' => $this->options['width'],
      '#states' => [
        // Show this number field only if a dialog is chosen above.
        'invisible' => [
          ':input[name="options[target]"]' => ['value' => ''],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    $account = \Drupal::currentUser();
    if ($empty && empty($this->options['bundle'])) {
      return [];
    }
    $bundle_type = $this->options['bundle'];
    $entity_type = $this->view->getBaseEntityType();
    // Try to be entity-type agnostic.
    if ($entity_type->id() === 'node') {
      // Link to add a node of the specified type, then return to our view.
      $url = Url::fromRoute('node.add', ['node_type' => $bundle_type], ['query' => $this->getDestinationArray()]);
      $access = $this->accessManager->checkNamedRoute('node.add', ['node_type' => $this->options['bundle']], $account);
    }
    else {
      // Try to get the entity creation path.
      $entity_links = $entity_type->get('links');
      if (isset($entity_links['add-form'])) {
        $path = str_replace('{' . $entity_type->id() . '}', $bundle_type, $entity_links['add-form']);
      }
      elseif (isset($entity_links['add-page'])) {
        $path = str_replace('{' . $entity_type->id() . '}', $bundle_type, $entity_links['add-page']);
      }
      if (empty($path)) {
        // An entity we don't know how to process, so exit.
        return;
      }
      $url = Url::fromUri($path);
      // Now use the URL to check access.
      $route_name = $url->getRouteName();
      $route_parameters = $url->getrouteParameters();
      $access = $this->accessManager->checkNamedRoute($route_name, $route_parameters, $account);
      // Finally, set a return path.
      $url->setOption('query', $this->getDestinationArray());
    }
    // Parse and sanitize provided classes.
    if ($this->options['class']) {
      $classes = explode(' ', $this->options['class']);
      foreach ($classes as $index => $class) {
        $classes[$index] = Html::getClass($class);
      }
    }
    else {
      $classes = [];
    }
    // Assembled elements into a link render array.
    $element = [
      '#type' => 'link',
      '#title' => $this->options['label'],
      '#url' => $url,
      '#options' => [
        'attributes' => ['class' => $classes],
      ],
      '#access' => $access,
    ];
    // Apply the selected dialog options.
    if ($this->options['target']) {
      $element['#options']['attributes']['class'][] = 'use-ajax';
      $element['#options']['attributes']['data-dialog-options'] = '{"width":500}';
      switch ($this->options['target']) {
        case 'tray':
          $element['#options']['attributes']['data-dialog-renderer'] = 'off_canvas';
          $element['#options']['attributes']['data-dialog-type'] = 'dialog';
          break;

        case 'modal':
          $element['#options']['attributes']['data-dialog-type'] = 'modal';
          break;
      }
    }
    return $element;
  }

}
