<?php

namespace Drupal\additional_js_css;

use Drupal\Core\File\FileSystemInterface;

/**
 * Provides helpers to manage additional JavaScript and CSS files.
 */
class AdditionalJSCSSManager implements AdditionalJSCSSManagerInterface {

  /**
   * The base directory for additional files.
   *
   * @var string
   */
  protected $additionalDirection;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new AdditionalJSCSSManager.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(FileSystemInterface $file_system) {
    $this->fileSystem = $file_system;
    $this->additionalDirection = 'public://additional_js_css/';
  }

  /**
   * {@inheritdoc}
   */
  public function getAdditionalFile(string $type) {
    if (in_array($type, ['js', 'css'])) {
      $type_path = $type == 'js' ? AdditionalJSCSSManagerInterface::JS_FILE_PATH : AdditionalJSCSSManagerInterface::CSS_FILE_PATH;
      $realpath = $this->fileSystem->realpath($type_path);
      if ($realpath && file_exists($realpath)) {
        $existing_content = file_get_contents($realpath);
        if ($existing_content !== FALSE) {
          return $existing_content;
        }
      }
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function saveAdditionalFile(string $type, string $value) {
    if (in_array($type, ['js', 'css'])) {
      if ($this->fileSystem->prepareDirectory($this->additionalDirection, FileSystemInterface::CREATE_DIRECTORY)) {
        $type_path = $type == 'js' ? AdditionalJSCSSManagerInterface::JS_FILE_PATH : AdditionalJSCSSManagerInterface::CSS_FILE_PATH;
        if ($this->fileSystem->saveData($value, $type_path, FileSystemInterface::EXISTS_REPLACE)) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

}
