<?php

namespace Drupal\address_decoupled\Plugin\rest\resource;

use Drupal\address_decoupled\Services\AddressDecoupledInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provide the Address decoupled Commerce Store supported countries resource.
 *
 * @RestResource(
 *   id = "address_decoupled_store_supported_countries",
 *   label = @Translation("Address decoupled - Store Supported Countries"),
 *   uri_paths = {
 *     "canonical" = "/address-decoupled/api/store/supported-countries/{store}",
 *   }
 * )
 */
class AddressDecoupledStoreSupportedCountriesResource extends ResourceBase {

  /**
   * The address decoupled.
   *
   * @var \Drupal\address_decoupled\Services\AddressDecoupledInterface
   */
  protected AddressDecoupledInterface $addressDecoupled;

  /**
   * Constructs a AddressDecoupledStoreSupportedCountriesResource object.
   *
   * @param array $config
   *   Config array which contains the information about the plugin instance.
   * @param string $module_id
   *   The module_id for the plugin instance.
   * @param mixed $module_definition
   *   The plugin implementation definition.
   * @param array $serializer_formats
   *   The available serialization formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\address_decoupled\Services\AddressDecoupledInterface $address_decoupled
   *   The address decoupled service.
   */
  public function __construct(
    array $config,
    $module_id,
    $module_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    AddressDecoupledInterface $address_decoupled
  ) {
    parent::__construct($config, $module_id, $module_definition, $serializer_formats, $logger);
    $this->addressDecoupled = $address_decoupled;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $config, $module_id, $module_definition) {
    return new static(
      $config,
      $module_id,
      $module_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('address_decoupled_rest'),
      $container->get('address_decoupled')
    );
  }

  /**
   * Get all the supported countries from a commmerce Store entity.
   *
   * @param int $store
   *   The commerce store entity ID.
   *
   * @return \Drupal\rest\ResourceResponse
   *   Returns the rest response with the store's supported countries data.
   */
  public function get(int $store): ResourceResponse {
    $supported_countries = $this->addressDecoupled->getStoreSupportedCountries($store);

    if (!$supported_countries) {
      return new ResourceResponse(['message' => 'There are no any supported countries for this store.']);
    }

    // Set the cache tag dependency.
    $response = new ResourceResponse($supported_countries);
    $response->addCacheableDependency(["commerce_store:$store"]);
    return $response;
  }

}
