<?php

namespace Drupal\address_js_geocoder\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\address\Plugin\Field\FieldWidget\AddressDefaultWidget;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\address_js_geocoder\Ajax\GeocodeAddressCommand;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use \Drupal\Component\Utility\NestedArray;


/**
 * Plugin implementation of the 'address_geocoder' widget.
 *
 * @FieldWidget(
 *   id = "address_geocoder",
 *   module = "address_js_geocoder",
 *   label = @Translation("Address geocoder"),
 *   field_types = {
 *     "address"
 *   }
 * )
 */
class AddressGeocoder extends AddressDefaultWidget implements ContainerFactoryPluginInterface
{

    /**
     * {@inheritdoc}
     */
    public static function defaultSettings()
    {
        return [
        'geofield_target' => '',
        ] + parent::defaultSettings();
    }

    /**
     * {@inheritdoc}
     */
    public function settingsForm(array $form, FormStateInterface $form_state)
    {
        $elements = [] + parent::settingsForm($form, $form_state);

        $elements['geofield_target'] = [
        '#type' => 'textfield',
        '#title' => t('Geofield target'),
        '#default_value' => $this->getSetting('geofield_target'),
        '#description' => t('The geofield field machine name that change this address will update.'),
        ];

        return $elements;
    }

    /**
     * {@inheritdoc}
     */
    public function settingsSummary()
    {
        $summary = [] + parent::settingsSummary();

        if (!empty($this->getSetting('geofield_target'))) {
            $summary[] = t('Changing this address will update: @geofield', ['@geofield' => $this->getSetting('geofield_target')]);
        }

        return $summary;
    }

    /**
     * {@inheritdoc}
     */
    public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state)
    {
        $element = parent::formElement($items, $delta, $element, $form, $form_state);
        //'#default_value' => isset($items[$delta]->value) ? $items[$delta]->value : NULL,

        $field_name = $this->fieldDefinition->getName();

        $form['#attached']['drupalSettings']['address_selector'] = 'edit-' . str_replace('_', "-", $field_name) . "-" . $delta;
        $form['#attached']['library'][] = 'address_js_geocoder/address_js_geocoder-library';


        $element['request_geocode'] = array(
        '#type' => 'button',
        '#value' => t('Get resolved addresses'),
        '#name' => "request_geocode",
        '#limit_validation_errors' => [],
        '#ajax' => array(
        'callback' => [$this, 'requestGeocodeCallback'],
        'wrapper' => 'geocoded-select-wrapper', // This element is updated with this AJAX callback.
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Requesting geocode...'),
        ],
        )
        );

        $element['geocoded_select'] = [
        '#type' => 'select',
        '#title' => $this->t('Resolved addresses options'),
        '#description' => $this->t("This option will be positioned at map below"),
        '#empty_value' => '',
        '#empty_option' => '- Select a value -',
        '#default_value' => '',
        '#prefix' => '<div id="geocoded-select-wrapper">',
        '#suffix' => '</div>',
        '#validated' => true,
        '#options' => [
        ],
        '#ajax' => [
        'callback' => [$this, 'geocodedSelectedCallback'], //alternative notation
        //'disable-refocus' => FALSE, // Or TRUE to prevent re-focusing on the triggering element.
        //'event' => 'change',
        'wrapper' => 'edit-output', // This element is updated with this AJAX callback.
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Showing in map...'),
        ],
        ],
        ];

        // Create a textbox that will be updatem
        // when the user selects an item from the select box above.
        $element['container']['resolved_address'] = [
        '#type' => 'hidden',
        '#size' => '60',
        '#disabled' => true,
        '#value' => '',
        ];

        $element['autogeocode'] = [
        '#type' => 'checkbox',
        '#title' => $this->
        t('Automatically position on map'),
        '#default_value' => true,
        ];


        return $element;
    }

    /**
     *  TODO WHENEVER it is implemented the autosubmit of address via fapi not
     *  jquery.
     *
     *  Adds ajax callback to all address fields.
     */
    public function addressMapTriggering(array $element, FormStateInterface &$form_state, $form)
    {
        // TODO WHENEVER it is implemented the autosubmit of address via FAPI not
        // jquery;
        $trigger_fields = ['address_line1', /*'address_line2',*/ 'postal_code', 'locality', 'country'];
        foreach ($trigger_fields as $trigger_field){
            $element[$trigger_field]['#ajax'] = [
            'callback' => [ 'Drupal\address_js_geocoder\Plugin\Field\FieldWidget\AddressGeocoder', 'autocheckCallback'],
            ];
        }
        return $element;
    }

    /**
     *  TODO WHENEVER it is implemented the autosubmit of address via fapi not
     *  jquery.
     */
    public function autocheckCallback(array &$form, FormStateInterface $form_state)
    {
        // It must fill the hidden field after that wait a timeout and
        // submit value to be geocoded... But address is multifield and so
        // on it's not easy to bind an event ti this callback.
        $field_name = $this->fieldDefinition->getName();
        $response = new AjaxResponse();
        $address = $form_state->getValue($field_name);

        $address_string = $this->addressToString($address);
        $form[$field_name]["widget"][0]['container']["resolved_address"]["value"] = $address_string;
        //TODO BELOW pass to data drupal selector  $form[$field_name]['#attributes']['data-drupal-selector'];
        $response->addCommand(new ReplaceCommand("#resolved_address-output", $form[$field_name]["widget"][0]['resolved_address']));
        return $response;

    }

    /**
     * Get the value from geocoded_select field and fill geofield target with the
     * selected coords.
     */
    public function geocodedSelectedCallback(array &$form, FormStateInterface $form_state)
    {
        $response = new AjaxResponse();
        $field_name = $this->fieldDefinition->getName();
        $triggering_element =  $form_state->getTriggeringElement();

        if ($selectedValue = NestedArray::getValue($form_state->getValues(), $triggering_element['#parents'])) {
            $coords_array = explode("|", $selectedValue);
            $coords = [
            "latitude" => $coords_array[0],
            "longitude" => $coords_array[1]
            ];

            $address_widget_selector_path = array_merge($triggering_element['#array_parents'], ['#attributes','data-drupal-selector']);
            $address_widget_selector = NestedArray::getValue($form, $address_widget_selector_path);

            $closer_parent_entity = $this->lookupCloserParentPath($triggering_element['#array_parents']);
            $target_geofield_selector_path =  array_merge($closer_parent_entity, [$this->getSetting('geofield_target'),'#attributes','data-drupal-selector']);
            $target_geofield_selector = NestedArray::getValue($form, $target_geofield_selector_path);
            //TODO : hem de passar field_name de field_adreca[0][value][lon] a particular_profiles[0][entity][field_adreca][0][value][lat]
            $closer_parent = $this->lookupCloserParentPath($triggering_element['#parents']);
            $address_field_name = $this->buildElementName($closer_parent, $this->field_name);
            $target_field_name = $this->buildElementName($closer_parent, $this->getSetting('geofield_target'));
            $response->addCommand(
                new GeocodeAddressCommand(
                    [
                    "widget" => $address_widget_selector,
                    "field_name" => $address_field_name
                    ], $coords, [
                    "widget" => $target_geofield_selector,
                    "field_name" => $target_field_name
                    ]
                )
            );
        }
        return $response;

    }

    /**
     * Gets the path to closer entity stack where the element is.
     * It works with array_parents and parents from triggering_element.
     * etc.tar.gz
     *
     * @param array $parents
     *   array_parents and parents from triggering_element.
     *
     * @return array
     *   an array with the path in form of array to closer parent.
     */
    public function lookupCloserParentPath($parents)
    {
        for($i = sizeof($parents) - 1; $i >= 0; $i--) {
            if ($parents[$i] != 'entity') {
                array_pop($parents);
            } else {
                return $parents;
            }
        }
        return $parents;
    }

    /**
     * Builds the element name that is used to get to element in js.
     *
     * @param array  $parents
     *   The parents of the element.
     * @param string $field_machine_name
     *   The field machine name of the element.
     *
     * @return string
     *   A string with the name to a form element in DOM.
     */
    public function buildElementName($parents, $field_machine_name)
    {
        if (count($parents)) {
            // As it comes from path from array build the name from array
            $top_name = array_shift($parents);
            $top_name .= '[' . implode('][', $parents) . ']';
            $top_name .= "[$field_machine_name]";
            return $top_name;
        } else {
            // As no parents array path use just the field_machine_name
            return $field_machine_name;
        }
    }



    public function validateForm(array &$form, FormStateInterface $form_state)
    {

        $triggering_element =  $form_state->getTriggeringElement();
        // TODO Check if geofield_target exists.
        $entity = $form_state->getFormObject()->getEntity();
        // check if the top entity in array_parents of this element has the address_field or we need to go
        // deeper (address field is in a referenced entity)
        $form_display = $form_state->getStorage()['form_display' ];
        // TODO add a way to obtain field definitions of the most closer parent as with the next logic cannot go further than
        //  parent[field_nested] -> entity -> field_address.
        $top_parent_entity_form_field_definitions = $form_display->get('fieldDefinitions')[$triggering_element['#array_parents'][0]];
        $top_parent_entity_form_settings = $top_parent_entity_form_field_definitions->getSettings();

        if ($top_parent_entity_form_field_definitions->getType() == 'entity_reference') {
            // TODO check for the containing type to not hardcode the bundle as the profile_type below
            // TODO ADDTESTS check between multiple types of parent entities (profile, media, group, etc)
            // TODO allow more nesting here (for example user->profile->media->field_adress)
            $top_parent_entity_fields = $entityFieldManager->getFieldDefinitions($top_parent_entity_form_settings['target_type'], array_keys($top_parent_entity_form_settings['handler_settings']['target_bundles'])[0]);
            $target_field_definition = $top_parent_entity_fields[$this->getSetting('geofield_target')];
        } elseif ($top_parent_entity_form_field_definitions->getType() == 'address') {
            // No nested entity, so we can use the fields of the entity belonging to the form directly:
            $target_field = $form_state->getFormObject()->getEntity()->getFields()[$this->getSetting('geofield_target')];
            if($target_field) {
                $target_field_definition = $target_field->getFieldDefinition();
            } else {
                $target_field_definition = false;
            }
        }

        if (!$target_field_definition) {
            // The array path to point to address field inputs elements
            $address_input_array_path =  array_slice($triggering_element['#parents'], 0, -2);
            $form_state->setErrorByName($triggering_element['#parents'], $this->t("The target @field field doesn't exist.", ['@field' => $this->getSetting('geofield_target')]));
            //The array path to point to select element
            $geocoded_select_array_path = array_slice($triggering_element['#array_parents'], 0, -1);
            $geocoded_select_array_path[] = 'geocoded_select';
            $geocoded_select = NestedArray::getValue($form, $geocoded_select_array_path);
            NestedArray::setValue(array_merge($geocoded_select_array_path, ['#options']), []);
            return  $geocoded_select;
        }
    }

    /**
     * An Ajax callback to request the geocode via geocode api.
     */
    public function requestGeocodeCallback(array &$form, FormStateInterface $form_state)
    {
        // Check if geofield_target exists
        $entity = $form_state->getFormObject()->getEntity();
        if ($form_state->getErrors()) {
            // If there are errors, we can show the form again with the errors in
            // the status_messages section.
            $response = new AjaxResponse();
            $form['status_messages'] = [
            '#type' => 'status_messages',
            '#weight' => -10,
            ];
            $response->addCommand(new OpenModalDialogCommand($this->t('Errors'), $form, ['width' => '50%',]));
            return $response;
        }

        $resolve_button = $form_state->getTriggeringElement();

        // The array path to point to address field inputs elements
        //$address_input_array_path[] = 'address';
        $address_input_array_path =  array_slice($resolve_button['#parents'], 0, -2);

        //The array path to point to select element
        $geocoded_select_array_path = array_slice($resolve_button['#array_parents'], 0, -1);
        $geocoded_select_array_path[] = 'geocoded_select';
        $geocoded_select = NestedArray::getValue($form, $geocoded_select_array_path);

        // Parents and array parents are distinct:
        // #array_parents contains the actual parent array keys of an element in a form structure.
        // #parents always contains the parent array keys of an element in the submitted form values.
        $parents = $resolve_button['#array_parents'];

        // check if the top entity in array_parents of this element has the address_field or we need to go
        // deeper (address field is in a referenced entity)
        $form_display = $form_state->getStorage()['form_display' ];
        // TODO add a way to obtain field definitions of the most closer parent as with the next logic cannot go further than
        //  parent[field_nested] -> entity -> field_address.
        $top_parent_entity_form_field_definitions = $form_display->get('fieldDefinitions')[$resolve_button['#array_parents'][0]];
        $top_parent_entity_form_settings = $top_parent_entity_form_field_definitions->getSettings();
        $entityFieldManager = \Drupal::service('entity_field.manager'); // TODO DI

        if ($top_parent_entity_form_field_definitions->getType() == 'entity_reference') {
            // TODO check for the containing type to not hardcode the bundle as the profile_type below
            // TODO ADDTESTS check between multiple types of parent entities (profile, media, group, etc)
            // TODO allow more nesting here (for example user->profile->media->field_adress)
            $top_parent_entity_fields = $entityFieldManager->getFieldDefinitions($top_parent_entity_form_settings['target_type'], array_keys($top_parent_entity_form_settings['handler_settings']['target_bundles'])[0]);
            $target_field_definition = $top_parent_entity_fields[$this->getSetting('geofield_target')];
        } elseif ($top_parent_entity_form_field_definitions->getType() == 'address') {
            // No nested entity, so we can use the fields of the entity belonging to the form directly:
            $target_field_definition = $form_state->getFormObject()->getEntity()->getFields()[$this->getSetting('geofield_target')]->getFieldDefinition();
        }
        if (!$target_field_definition) {
            $response = new AjaxResponse();
            $message = $this->t("The target '@field' field doesn't exist. Contact administrator.", ['@field' => $this->getSetting('geofield_target')]);
            $response->addCommand(new OpenModalDialogCommand($this->t('Errors'), $message, ['width' => '50%', 'minHeight' => 200, 'resizable' => true]));
            return $response;
        }

        // Check if target field has geocode enabled
        /**
       * @var \Drupal\geocoder_field\GeocoderFieldPluginInterface $field_plugin 
*/ //TODO DI!
        if (!$target_field_plugin = \Drupal::service('geocoder_field.plugin.manager.field')->getPluginByFieldType($target_field_definition->getType())) {
            // There's no geocoding field plugin to handle this type of field.
            $response = new AjaxResponse();
            $message = $this->t("There's no geocoding field plugin in '@field' to handle this type of field. Contact administrator.", ['@field' => $this->getSetting('geofield_target')]);
            $response->addCommand(new OpenModalDialogCommand($this->t('Errors'), $message, ['width' => '50%', 'minHeight' => 200, 'resizable' => true]));
            return $response;
        }
        $target_field_config = $target_field_definition->getThirdPartySettings('geocoder_field');
        $providers_ids = $target_field_definition->getThirdPartySettings('geocoder_field')["providers"];
        $providers = \Drupal::entityTypeManager()->getStorage('geocoder_provider')->loadMultiple($providers_ids);

        // Prepare our textfield. check if the example select field has a selected option.
        if ($address =  NestedArray::getValue($form_state->getUserInput(), $address_input_array_path)) {

            $address_string = $this->addressToString($address);
            // Geocode the address:
            $addressCollection = \Drupal::service('geocoder')->geocode($address_string, $providers);

            // Set select options from results!!!
            $geocode_opts = $addressCollection;

            // Reset the options.
            $geocoded_select['#options'] = [];

            // Fill the select options
            foreach($geocode_opts as $key => $val){
                $geocoded_select['#options'][ $val->getCoordinates()->getLatitude() ."|". $val->getCoordinates()->getLongitude()] = $val->getFormattedAddress();
            }
        }


        $response = new AjaxResponse();
        // return commands Replace the options from search
        // and trigger select change event to populate the map
        $response->addCommand(new ReplaceCommand("#geocoded-select-wrapper", $geocoded_select));
        $response->addCommand(new InvokeCommand('#geocoded-select-wrapper select', 'trigger', ['change']));

        return $response;
    }

    /**
     * Sets the address from Adress element type to string
     *
     * @param array $address_fields
     *   The address fields from address->value
     */
    protected function addressToString(array $address_fields)
    {

        $addresses = [];
        foreach($address_fields as $delta => $item){
            $address[] = !empty($item["address"]['address_line1']) ? $item["address"]['address_line1'] : null;
            $address[] = !empty($item["address"]['address_line2']) ? $item["address"]['address_line2'] : null;
            $address[] = !empty($item["address"]['postal_code']) ? $item["address"]['postal_code'] : null;
            $address[] = !empty($item["address"]['locality']) ? $item["address"]['locality'] : null;
            $address[] = !empty($item["address"]['country']) ? $item["address"]['country'] : null;
            $addresses[] = implode(' ', array_filter($address));
        }
        //TODO allow multiple address lookup By now only first...
        return $addresses[0];

    }

}
