<?php

namespace Drupal\address_processor\Plugin\search_api\processor;

use Drupal\address\Plugin\Field\FieldType\AddressFieldItemList;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Locale\CountryManagerInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\SearchApiException;
use Drupal\search_api\Utility\DataTypeHelperInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This processor adds country name.
 *
 * @SearchApiProcessor(
 *   id = "address_processor",
 *   label = @Translation("Address processor"),
 *   description = @Translation("Add processor to address to add allow search country name."),
 *   stages = {
 *     "preprocess_index" = -50,
 *   }
 * )
 */
class AddressProcessor extends ProcessorPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * The data type helper.
   *
   * @var \Drupal\search_api\Utility\DataTypeHelperInterface|null
   */
  protected $dataTypeHelper;

  /**
   * The country manager.
   *
   * @var \Drupal\Core\Locale\CountryManagerInterface
   */
  protected $countryManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $processor */
    $processor = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $processor->setCountryManager($container->get('country_manager'));

    return $processor;
  }

  /**
   * Retrieves the country manager.
   *
   * @return \Drupal\Core\Locale\CountryManagerInterface
   *   The country manager.
   */
  public function getCountryManager() {
    return $this->countryManager;
  }

  /**
   * Sets the country manager.
   *
   * @param \Drupal\Core\Locale\CountryManagerInterface $countryManager
   *   The new country manager.
   *
   * @return $this
   */
  public function setCountryManager(CountryManagerInterface $countryManager) {
    $this->countryManager = $countryManager;
    return $this;
  }

  /**
   * Retrieves the data type helper.
   *
   * @return \Drupal\search_api\Utility\DataTypeHelperInterface
   *   The data type helper.
   */
  public function getDataTypeHelper() {
    return $this->dataTypeHelper ?: \Drupal::service('search_api.data_type_helper');
  }

  /**
   * Sets the data type helper.
   *
   * @param \Drupal\search_api\Utility\DataTypeHelperInterface $data_type_helper
   *   The new data type helper.
   *
   * @return $this
   */
  public function setDataTypeHelper(DataTypeHelperInterface $data_type_helper) {
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function preprocessIndexItems(array $items) {
    $countries = $this->countryManager->getStandardList();

    foreach ($items as $item) {
      try {
        $entity = $item->getOriginalObject()->getValue();
      }
      catch (SearchApiException $e) {
        // Skip this item if there's an exception.
        continue;
      }

      if (!$entity instanceof ContentEntityInterface) {
        // Skip non-content entities.
        continue;
      }

      foreach ($item->getFields() as $field) {
        if (!in_array($field->getType(), ['text', 'string'], TRUE)) {
          // Process only 'text' or 'string' fields.
          continue;
        }

        $field_name = $field->getPropertyPath();
        if (!$entity->hasField($field_name)) {
          // Skip if the entity lacks the field.
          continue;
        }

        $field_value = $entity->get($field_name);
        if ($field_value->isEmpty() || !$field_value instanceof AddressFieldItemList) {
          // Skip if the field is empty or not an address field.
          continue;
        }

        foreach ($field_value->getValue() as $address) {
          $country_code = $address['country_code'] ?? NULL;
          if (!$country_code || !isset($countries[$country_code])) {
            // Skip if no country code or invalid code.
            continue;
          }

          $country = $countries[$country_code];
          if ($country instanceof TranslatableMarkup) {
            $country = $country->getUntranslatedString();
          }

          $field->addValue($country);
        }
      }
    }
  }

}
