<?php

namespace Drupal\address_suggestion\Plugin\Field\FieldWidget;

use CommerceGuys\Addressing\Country\CountryRepositoryInterface;
use Drupal\address\Plugin\Field\FieldWidget\AddressDefaultWidget;
use Drupal\address_suggestion\AddressProviderManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldConfigInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Plugin implementation of the 'address_autocomplete' widget.
 *
 * @FieldWidget(
 *   id = "address_suggestion",
 *   label = @Translation("Address suggestion"),
 *   field_types = {
 *     "address"
 *   }
 * )
 */
class AddressSuggestionWidget extends AddressDefaultWidget {

  /**
   * Address provider plugin manager.
   *
   * @var \Drupal\address_suggestion\AddressProviderManager
   */
  protected $addressProvider;

  /**
   * Constructs a WidgetBase object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \CommerceGuys\Addressing\Country\CountryRepositoryInterface $country_repository
   *   The country repository.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\address_suggestion\AddressProviderManager $plugin_address_provider
   *   The plugin address provider manager.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, CountryRepositoryInterface $country_repository, EventDispatcherInterface $event_dispatcher, ConfigFactoryInterface $config_factory, AddressProviderManager $plugin_address_provider) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $country_repository, $event_dispatcher, $config_factory);
    $this->addressProvider = $plugin_address_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($plugin_id, $plugin_definition,
      $configuration['field_definition'], $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('address.country_repository'),
      $container->get('event_dispatcher'),
      $container->get('config.factory'),
      $container->get('plugin.manager.address_provider'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $element['address']['#type'] = 'address_suggestion';
    $fieldDefinitions = $items->getFieldDefinition();
    $element['address']['#entity_type'] = $fieldDefinitions->getTargetEntityTypeId();
    $element['address']['#bundle'] = $fieldDefinitions->getTargetBundle();
    $element['address']['#field_name'] = $fieldDefinitions->getName();
    $settings = $this->getSettings();
    if (!empty($this->getSetting('location_field'))) {
      $entity_type = $this->fieldDefinition->getTargetEntityTypeId();
      $bundle = $this->fieldDefinition->getTargetBundle();
      $entityFieldManager = \Drupal::service('entity_field.manager');
      $fieldDefinitions = $entityFieldManager->getFieldDefinitions($entity_type, $bundle);
      $field_name = $this->getSetting('location_field');
      $settings['type_field'] = $fieldDefinitions[$field_name]->getType();
    }
    $form['#attached']['drupalSettings']['address_suggestion'] = $settings;

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'location_field' => '',
      'provider' => '',
      'endpoint' => '',
      'api_key' => '',
      'username' => '',
      'password' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $options = [];
    $entity_type = $form["#entity_type"];
    $bundle = $form["#bundle"];
    $entityFieldManager = \Drupal::service('entity_field.manager');
    $fieldDefinitions = $entityFieldManager->getFieldDefinitions($entity_type, $bundle);
    $userInput = $form_state->getUserInput();
    $typSupport = ['geofield_latlon', 'geolocation_latlng'];
    if (!empty($userInput["fields"])) {
      foreach ($userInput["fields"] as $field_name => $field_widget) {
        if (!empty($field_widget['type']) &&
          in_array($field_widget['type'], $typSupport)) {
          $options[$field_name] = (string) $fieldDefinitions[$field_name]->getLabel();
        }
      }
    }
    $fieldName = $this->fieldDefinition->getName();
    $moduleSupport = ['geolocation', 'geofield'];
    if (empty($options)) {
      foreach ($fieldDefinitions as $field_name => $field_definition) {
        if ($field_definition instanceof FieldConfigInterface &&
          in_array($field_definition->getType(), $moduleSupport)) {
          $options[$field_name] = (string) $field_definition->getLabel();
        }
      }
    }
    if (!empty($options)) {
      $element['location_field'] = [
        '#type' => 'select',
        '#title' => $this->t('Location field'),
        '#default_value' => $this->getSetting('location_field'),
        '#options' => $options,
        "#empty_option" => $this->t('- Select field -'),
        "#description" => $this->t('You can attach a location field to get the coordinates'),
      ];
    }
    $endPointUrl = [
      $this->t("Fill in if you want to use your own api"),
    ];
    $options = [];
    $key = [];
    $state = [];
    $login = [];
    $addressProvider = $this->addressProvider->getDefinitions();
    foreach ($addressProvider as $provider) {
      $options[$provider['id']] = $provider['label'];
      if (!empty($provider['api'])) {
        $endPointUrl[] = $provider['label'] . ': ' . $provider['api'];
        if (!isset($provider['nokey'])) {
          $key[] = $provider['label'];
          $state[] = ['value' => $provider['id']];
        }
        if (isset($provider['login'])) {
          $login[] = ['value' => $provider['id']];
        }
      }
    }
    $element['provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Provider'),
      '#default_value' => $this->getSetting('provider'),
      '#options' => $options,
      "#empty_option" => $this->t('- Select provider -'),
    ];
    $element['endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('Custom API Address endpoint'),
      '#default_value' => $this->getSetting('endpoint'),
      "#description" => implode('<br/>', $endPointUrl),
    ];
    $element['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $this->getSetting('api_key'),
      "#description" => $this->t('Required for provider: @key', ['@key' => implode(', ', $key)]),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $fieldName . '][settings_edit_form][settings][provider]"]' => [
            $state,
          ],
        ],
      ],
    ];
    $element['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('post.ch API username'),
      '#default_value' => $this->getSetting('username'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $fieldName . '][settings_edit_form][settings][provider]"]' => [$login],
        ],
      ],
    ];
    $element['password'] = [
      '#type' => 'password',
      '#title' => $this->t('post.ch API password'),
      '#default_value' => $this->getSetting('password'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $fieldName . '][settings_edit_form][settings][provider]"]' => [$login],
        ],
      ],
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    if (!empty($this->getSetting('location_field'))) {
      $summary[] = $this->t('Location field: @field', ['@field' => $this->getSetting('location_field')]);
    }
    if (!empty($provider = $this->getSetting('provider'))) {
      $provider = ucfirst(str_replace('_', ' ', $provider));
      $summary[] = $this->t('Provider: @provider', ['@provider' => $provider]);
    }
    return $summary;
  }

}
