<?php

namespace Drupal\addtoany;

use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Class for updating local scripts.
 */
class UpdateScripts {

  /**
   * Path to contract files.
   */
  const SCRIPTSPATH = 'public://js/addtoany/menu/';

  /**
   * List of the latest scripts to download.
   */
  const LATESTSCRIPTS = 'https://www.addtoany.com/ext/updater/files_list/';

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a UpdateScripts object.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file URL generator.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger channel factory service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state key/value store.
   */
  public function __construct(FileSystemInterface $fileSystem, FileUrlGeneratorInterface $fileUrlGenerator, LoggerChannelFactoryInterface $logger, StateInterface $state) {
    $this->fileSystem = $fileSystem;
    $this->logger = $logger->get('addtoany');
    $this->fileUrlGenerator = $fileUrlGenerator;
    $this->state = $state;
  }

  /**
   * Download the latest scripts.
   *
   * @see https://www.addtoany.com/ext/updater/files_list
   *
   * @return false|void
   *   Returns FALSE on failure otherwise nothing.
   */
  public function updateLatestScripts() {
    try {
      // List of files that will be downloaded with version.
      $files_to_download = $this->getScripts();

      // Create directory for output.
      $directory = self::SCRIPTSPATH . $files_to_download['version'];
      // Now we only need the files that will be downloaded.
      $latest_version = $files_to_download['version'];
      unset($files_to_download['version']);
      // Empty the directory before putting new content.
      // $this->fileSystem->deleteRecursive($directory);
      if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        throw new \RuntimeException("Failed to create the directory: " . $directory);
      }

      foreach ($files_to_download as $file_to_download) {
        $content = file_get_contents($file_to_download);
        $name = basename($file_to_download);
        $file = $directory . '/' . $name;
        // Remove hardcoded external link in page.js that downloads information
        // document.
        if ($name == 'page.js') {
          preg_match('#https?://[^,\s()<>]+(?:\([\w\d]+\)|([^,[:punct:]\s])(.html)[-A-Za-z0-9+&@\#/%?=~_|!:,.;]+[-A-Za-z0-9+&@\#/%=~_|])#', $content, $match);
          $content = str_replace($match[0], '', $content);
        }
        file_put_contents($file, $content);
      }
      $this->state->set('addtoany.scripts_version', $latest_version);
    }
    catch (\Exception $e) {
      $this->logger->error(
        $e->getMessage() . "\n Failed to update addtoany scripts\n",
      );
      return FALSE;
    }
  }

  /**
   * Returns a list of the latest script files.
   *
   * Use the hash in the file names for versioning.
   *
   * @return array|false
   *   Array of file to download or FALSE.
   */
  public function getScripts() {
    try {
      $files_to_download = file_get_contents(self::LATESTSCRIPTS);
      $files_to_download = explode("\r\n", $files_to_download, 20);
      // Take the first file and extract
      // the hash in the name and use it for versioning.
      $first_file = reset($files_to_download);
      $file_name_parts = explode('.', basename($first_file));
      $version_hash = $file_name_parts[1];
      if (empty($version_hash)) {
        throw new \RuntimeException("Failed to get the hash from js file name: " . basename($first_file));
      }
      $files_to_download['version'] = $version_hash;
      return $files_to_download;
    }
    catch (\Exception $e) {
      $this->logger->error(
        $e->getMessage() . "\n Failed to get the scripts\n",
      );
      return FALSE;
    }
  }

  /**
   * Remove all the versions of the scripts except the latest version.
   *
   * Look up getScripts() to check how versioning is done.
   */
  public function cleanup() {
    try {
      $all_script_versions = $this->fileSystem->scanDirectory(self::SCRIPTSPATH, '/.*/', ['recurse' => FALSE]);
      $latest_version = $this->state->get('addtoany.scripts_version');
      if (isset($all_script_versions)) {
        foreach ($all_script_versions as $script_version) {
          if ($script_version->name === $latest_version) {
            continue;
          }
          $this->fileSystem->deleteRecursive($script_version->uri);
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error(
        $e->getMessage() . "\n Failed to cleanup old scripts\n",
      );
    }
  }

}
