<?php

namespace Drupal\addtocal_augment\Plugin\DateAugmenter;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\date_augmenter\DateAugmenter\DateAugmenterPluginBase;
use Drupal\date_augmenter\Plugin\PluginFormTrait;

/**
 * Date Augmenter plugin to inject Add to Calendar links.
 *
 * @DateAugmenter(
 *   id = "addtocal",
 *   label = @Translation("Add To Calendar Links"),
 *   description = @Translation("Adds links to add an events dates to a user's preferred calendar."),
 *   weight = 0
 * )
 */
class AddToCal extends DateAugmenterPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  protected $processService;
  protected $config;
  protected $output;

  // /**
  //  * {@inheritdoc}
  //  */
  // public function settings($setting_key = NULL) {
  //   if (!is_null($setting_key)) {
  //     return $this->pluginDefinition['settings'][$setting_key] ?? $this->pluginDefinition['settings'];
  //   }
  //   return $this->pluginDefinition['settings'];
  // }

  /**
   * Builds and returns a render array for the task.
   *
   * @param array $output
   *   The existing render array, to be augmented, passed by reference.
   * @param Drupal\Core\Datetime\DrupalDateTime $start
   *   The object which contains the start time.
   * @param Drupal\Core\Datetime\DrupalDateTime $end
   *   The optionalobject which contains the end time.
   * @param array $options
   *   An array of options to further guide output.
   */
  public function augmentOutput(array &$output, DrupalDateTime $start, DrupalDateTime $end = NULL, array $options = []) {
    // Use provided settings if they exist, otherwise look for plugin config.
    $config = $options['settings'] ?? $this->getConfiguration();
    if (empty($config['title']) && !isset($options['entity'])) {
      // TODO: log some kind of warning that we can't work without the entity
      // or a provided title?
      return;
    }
    $end_fallback = $end ?? $start;
    $now = new DrupalDateTIme();
    if ($end_fallback < $now && !$config['past_events']) {
      return;
    }
    $entity = $options['entity'] ?? NULL;
    if (!$end) {
      $end = $start;
    }
    $timezone = $options['timezone'] ?? NULL;
    if (isset($options['allday']) && $options['allday']) {
      $start_formatted = $start->format("Ymd", $timezone);
      // Offset the end by one day for calendar ingestion.
      $end->add(new \DateInterval('P1D'));
      $end_formatted = $end->format("Ymd", $timezone);
    }
    else {
      // TODO: support native timezone, if set?
      $start_formatted = $start->format("Ymd\\THi00\\Z", $timezone);
      $end_formatted = $end->format("Ymd\\THi00\\Z", $timezone);
    }
    if (!empty($config['title'])) {
      $label = $this->parseField($config['title'], $entity);
    }
    else {
      $label = $entity->label();
    }
    $description = NULL;
    if (!empty($config['description'])) {
      $description = $this->parseField($config['description'], $entity, TRUE);
    }
    $location = NULL;
    if (!empty($config['location'])) {
      $location = $this->parseField($config['location'], $entity, TRUE);
    }
    $ical_link = 'data:text/calendar;charset=utf8,BEGIN:VCALENDAR%0AVERSION:2.0%0ABEGIN:VEVENT'
                 // TODO: Format with timezone if set, instead of UTC?
                 . '%0ADTSTART:' . $start_formatted
                 . '%0ADTEND:' . $end_formatted
                 . '%0ASUMMARY:' . $label;
    if ($description) {
      $ical_link .= '%0ADESCRIPTION:' . $description;
    }
    if ($location) {
      $ical_link .= '%0ALOCATION:' . $location;
    }
    $ical_link .= '%0AEND:VEVENT%0AEND:VCALENDAR';

    $google_link = 'https://www.google.com/calendar/r/eventedit?text='
                   . $label
                   . '&dates=' . $start_formatted . '/' . $end_formatted;
    if ($description) {
      $google_link .= '&details=' . $description;
    }
    if ($location) {
      $google_link .= '&location=' . $location;
    }

    $output['addtocal'] = [
      '#theme' => 'addtocal_links',
      '#google' => $google_link,
      '#ical' => $ical_link,
    ];
  }

  public function parseField($field_value, $entity, $strip_markup = FALSE) {
    if (\Drupal::hasService('token') && $entity) {
      $token_service = \Drupal::service('token');
      $token_data = [
        $entity->getEntityTypeId() => $entity,
      ];
      $field_value = $token_service->replace($field_value, $token_data);
    }
    if ($strip_markup) {
      // Strip tags.
      $field_value = strip_tags($field_value);

      // Strip out line breaks.
      $field_value = preg_replace('/\n|\r|\t/m', ' ', $field_value);

      // Strip out non-breaking spaces.
      $field_value = str_replace('&nbsp;', ' ', $field_value);
      $field_value = str_replace("\xc2\xa0", ' ', $field_value);

      // Strip out extra spaces.
      $field_value = trim(preg_replace('/\s\s+/', ' ', $field_value));
    }
    return $field_value;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'event_title' => '',
      'location' => '',
      'description' => '',
      'past_events' => FALSE,
    ];
  }

  /**
   * Create configuration fields for the plugin form, or injected directly.
   *
   * @param array $form
   *   The form array.
   * @param array $settings
   *   The setting to use as defaults.
   * @param mixed $field_definition
   *   A parameter to define the field being modified. Likely FieldConfig.
   *
   * @return array
   *   The updated form array.
   */
  public function configurationFields(array $form, ?array $settings, $field_definition) {
    if (empty($settings)) {
      $settings = $this->defaultConfiguration();
    }
    $form['event_title'] = [
      '#title' => $this->t('Event title'),
      '#type' => 'textfield',
      '#default_value' => $settings['event_title'],
      '#description' => $this->t('Optional - if left empty, the entity label will be used. You can use static text or tokens.'),
    ];

    $form['location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Optional. You can use static text or tokens.'),
      '#default_value' => $settings['location'],
    ];

    $form['description'] = [
      '#title' => $this->t('Event description'),
      '#type' => 'textarea',
      '#default_value' => $settings['description'],
      '#description' => $this->t('Optional. You can use static text or tokens.'),
    ];

    $form['past_events'] = [
      '#title' => $this->t('Show Add to Cal widget for past events?'),
      '#type' => 'checkbox',
      '#default_value' => $settings['past_events'],
    ];

    if (function_exists('token_theme')) {
      $type = NULL;
      if (method_exists($field_definition, 'getTargetEntityTypeId')) {
        $type = $field_definition->getTargetEntityTypeId();
      }
      // TODO: support other field types?
      $form['token_tree_link'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => [$type],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $this->configurationFields($form, $this->configuration);

    return $form;
  }

}
