<?php

namespace Drupal\addtocal_augment\Plugin\DateAugmenter;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\date_augmenter\DateAugmenter\DateAugmenterPluginBase;
use Drupal\date_augmenter\Plugin\PluginFormTrait;

/**
 * Date Augmenter plugin to inject Add to Calendar links.
 *
 * @DateAugmenter(
 *   id = "addtocal",
 *   label = @Translation("Add To Calendar Links"),
 *   description = @Translation("Adds links to add an events dates to a user's preferred calendar."),
 *   weight = 0
 * )
 */
class AddToCal extends DateAugmenterPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  protected $processService;
  protected $config;
  protected $output;

  /**
   * Builds and returns a render array for the task.
   *
   * @param array $output
   *   The existing render array, to be augmented, passed by reference.
   * @param Drupal\Core\Datetime\DrupalDateTime $start
   *   The object which contains the start time.
   * @param Drupal\Core\Datetime\DrupalDateTime $end
   *   The optionalobject which contains the end time.
   * @param array $options
   *   An array of options to further guide output.
   */
  public function augmentOutput(array &$output, DrupalDateTime $start, DrupalDateTime $end = NULL, array $options = []) {
    // Use provided settings if they exist, otherwise look for plugin config.
    $config = $options['settings'] ?? $this->getConfiguration();
    if (empty($config['title']) && !isset($options['entity'])) {
      // TODO: log some kind of warning that we can't work without the entity
      // or a provided title?
      return;
    }
    $end_fallback = $end ?? $start;
    $now = new DrupalDateTIme();
    if ($end_fallback < $now && !$config['past_events']) {
      return;
    }
    $entity = $options['entity'] ?? NULL;
    if (!$end) {
      $end = $start;
    }
    $timezone = $options['timezone'] ?? NULL;
    if (isset($options['allday']) && $options['allday']) {
      $start_formatted = $start->format("Ymd", $timezone);
      // Offset the end by one day for calendar ingestion.
      $end->add(new \DateInterval('P1D'));
      $end_formatted = $end->format("Ymd", $timezone);
    }
    else {
      // TODO: Format with timezone if set, instead of UTC?
      $start_formatted = $start->format("Ymd\\THi00\\Z", $timezone);
      $end_formatted = $end->format("Ymd\\THi00\\Z", $timezone);
    }
    if (!empty($config['title'])) {
      $label = $this->parseField($config['title'], $entity);
    }
    else {
      $label = $this->parseField($entity->label(), FALSE);
    }
    $description = NULL;
    if (!empty($config['description'])) {
      $description = $this->parseField($config['description'], $entity, TRUE);
      $max_length = isset($config['max_desc']) ? $config['max_desc'] : 60;
      if ($max_length) {
        // TODO: Use Smart Trim if available.
        // TODO: Make the use of ellipsis configurable?
        $description = trim(substr($description, 0, $max_length)) . '...';
      }
    }
    $location = NULL;
    if (!empty($config['location'])) {
      $location = $this->parseField($config['location'], $entity, TRUE);
    }
    $ical_link = 'data:text/calendar;charset=utf8,BEGIN:VCALENDAR%0AVERSION:2.0%0ABEGIN:VEVENT'
                 . '%0ADTSTART:' . $start_formatted
                 . '%0ADTEND:' . $end_formatted
                 . '%0ASUMMARY:' . $label;
    if ($description) {
      $ical_link .= '%0ADESCRIPTION:' . $description;
    }
    if ($location) {
      $ical_link .= '%0ALOCATION:' . $location;
    }
    $ical_link .= '%0AEND:VEVENT%0AEND:VCALENDAR';

    $google_link = 'https://www.google.com/calendar/r/eventedit?text='
                   . $label
                   . '&dates=' . $start_formatted . '/' . $end_formatted;
    if ($description) {
      $google_link .= '&details=' . $description;
    }
    if ($location) {
      $google_link .= '&location=' . $location;
    }

    $output['addtocal'] = [
      '#theme' => 'addtocal_links',
      '#google' => $google_link,
      '#ical' => $ical_link,
    ];
  }

  /**
   * Manipulate the provided value, checking for tokens and cleaning up.
   *
   * @param string $field_value
   *   The value to manipulate.
   * @param mixed $entity
   *   The entity whose values can be used to replace tokens.
   * @param bool $strip_markup
   *   Whether or not to clean up the output.
   *
   * @return string
   *   The manipulated value, prepared for use in a link href.
   */
  public function parseField($field_value, $entity, $strip_markup = FALSE) {
    if (\Drupal::hasService('token') && $entity) {
      $token_service = \Drupal::service('token');
      $token_data = [
        $entity->getEntityTypeId() => $entity,
      ];
      $field_value = $token_service->replace($field_value, $token_data);
    }
    if ($strip_markup) {
      // Strip tags. Requires decoding entities, which will be re-encoded later.
      $field_value = strip_tags(html_entity_decode($field_value));

      // Strip out line breaks.
      $field_value = preg_replace('/\n|\r|\t/m', ' ', $field_value);

      // Strip out non-breaking spaces.
      $field_value = str_replace('&nbsp;', ' ', $field_value);
      $field_value = str_replace("\xc2\xa0", ' ', $field_value);

      // Strip out extra spaces.
      $field_value = trim(preg_replace('/\s\s+/', ' ', $field_value));
    }
    // Return the value escaped for output in a URL.
    return htmlentities(urlencode($field_value));
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'event_title' => '',
      'location' => '',
      'description' => '',
      'max_desc' => 60,
      'past_events' => FALSE,
    ];
  }

  /**
   * Create configuration fields for the plugin form, or injected directly.
   *
   * @param array $form
   *   The form array.
   * @param array|null $settings
   *   The setting to use as defaults.
   * @param mixed $field_definition
   *   A parameter to define the field being modified. Likely FieldConfig.
   *
   * @return array
   *   The updated form array.
   */
  public function configurationFields(array $form, ?array $settings, $field_definition) {
    if (empty($settings)) {
      $settings = $this->defaultConfiguration();
    }
    $form['event_title'] = [
      '#title' => $this->t('Event title'),
      '#type' => 'textfield',
      '#default_value' => $settings['event_title'],
      '#description' => $this->t('Optional - if left empty, the entity label will be used. You can use static text or tokens.'),
    ];

    $form['location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Optional. You can use static text or tokens.'),
      '#default_value' => $settings['location'],
    ];

    $form['description'] = [
      '#title' => $this->t('Event description'),
      '#type' => 'textarea',
      '#default_value' => $settings['description'],
      '#description' => $this->t('Optional. You can use static text or tokens.'),
    ];

    $form['max_desc'] = [
      '#title' => $this->t('Maximum description length'),
      '#type' => 'number',
      '#default_value' => isset($settings['max_desc']) ? $settings['max_desc'] : 60,
      '#description' => $this->t('Trim the desctiption to a specified length. Leave empty or use zero to not trim the value.'),
    ];

    $form['past_events'] = [
      '#title' => $this->t('Show Add to Cal widget for past events?'),
      '#type' => 'checkbox',
      '#default_value' => $settings['past_events'],
    ];

    if (function_exists('token_theme')) {
      $type = NULL;
      if (method_exists($field_definition, 'getTargetEntityTypeId')) {
        $type = $field_definition->getTargetEntityTypeId();
      }
      // TODO: support other field types?
      $form['token_tree_link'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => [$type],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $this->configurationFields($form, $this->configuration);

    return $form;
  }

}
