<?php

namespace Drupal\admin_password_reset\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Form for sending password reset links to users.
 */
class SendPasswordResetForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new SendPasswordResetForm.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    MessengerInterface $messenger,
    LoggerInterface $logger
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->messenger = $messenger;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('messenger'),
      $container->get('logger.factory')->get('admin_password_reset')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'admin_password_reset_send_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $user = NULL) {
    $form['description'] = [
      '#type' => 'markup',
      '#markup' => $this->t('Send a password reset link to %username (%email).', [
        '%username' => $user->getAccountName(),
        '%email' => $user->getEmail(),
      ]),
    ];

    $form['block_account'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Block account until password is reset'),
      '#description' => $this->t('If checked, the account will be blocked until the user resets their password.'),
      '#default_value' => FALSE,
    ];

    $form['user_id'] = [
      '#type' => 'value',
      '#value' => $user->id(),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send Reset Link'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $user_id = $form_state->getValue('user_id');
    $block_account = $form_state->getValue('block_account');

    /** @var \Drupal\user\UserInterface $account */
    $account = $this->entityTypeManager->getStorage('user')->load($user_id);

    if ($block_account) {
      $account->block();
      $account->save();
    }

    // Send password reset email
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    $mail = _user_mail_notify('password_reset', $account, $langcode);

    if (!empty($mail)) {
      $this->logger->notice('Password reset instructions mailed to %name at %email.', [
        '%name' => $account->getAccountName(),
        '%email' => $account->getEmail(),
      ]);
      $this->messenger->addStatus($this->t('Password reset instructions have been sent to %email.', [
        '%email' => $account->getEmail(),
      ]));
      if ($block_account) {
        $this->messenger->addStatus($this->t('The account has been blocked until the password is reset.'));
      }
    }
    else {
      $this->messenger->addError($this->t('There was a problem sending the password reset email.'));
    }
  }

}
