<?php

namespace Drupal\admin_status_report\EventSubscriber;

use Drupal\admin_status_report\AdminStatusPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;

/**
 * Class AdminStatusEventSubscriber.
 *
 * This class catches kernel.request events and displays messages from enabled
 * plugins.
 *
 * @package Drupal\admin_status_report
 */
class AdminStatusEventSubscriber implements EventSubscriberInterface {

  /**
   * Drupal\admin_status_report\AdminStatusPluginManager definition.
   *
   * @var \Drupal\admin_status_report\AdminStatusPluginManager
   */
  protected $adminStatusManager;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The Configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * Construct an AdminStatusEventSubscriber object.
   *
   * @param \Drupal\admin_status_report\AdminStatusPluginManager $plugin_manager_admin_status
   *   The Admin Status Plugin Manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   */
  public function __construct(AdminStatusPluginManager $plugin_manager_admin_status, RendererInterface $renderer, MessengerInterface $messenger, ConfigFactoryInterface $config_factory) {
    $this->adminStatusManager = $plugin_manager_admin_status;
    $this->renderer = $renderer;
    $this->messenger = $messenger;
    $this->config = $config_factory->get('admin_status_report.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Tell Symfony that we want to hear about kernel.request events.
    $events['kernel.request'] = ['kernelRequest'];
    return $events;
  }

  /**
   * Handles kernel.request events.
   *
   * @param Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The Symfony event.
   */
  public function kernelRequest(RequestEvent $event) {
    static $renderer;

    // Get our saved config data.
    $config = $this->config;
    $plugin_status = $config->get('plugin_status');

    if (!is_array($plugin_status)) {
      // Handle the case where $plugin_status is not set or is not an array.
      $plugin_status = [];
    }

    // Run through each plugin and invoke its message and status methods if it's
    // enabled.
    foreach ($plugin_status as $plugin_id => $status) {
      if ($status['enabled']) {
        // Create the message with the configuration values.
        $plugin = $this->adminStatusManager->createInstance(
        $plugin_id, ['of' => 'configuration values']
        );
        $configValues = empty($status['config']) ? [] : $status['config'];
        $messages = $plugin->message($configValues);
        if (isset($messages['status'])) {
          $messages = [$messages];
        }

        foreach ($messages as $message) {
          // If the plugin returns an empty message, there is nothing to
          // display. Otherwise, get the status and display the message.
          if (!empty($message)) {
            $msgText = $message['message'];
            if (is_array($msgText)) {
              if (empty($renderer)) {
                $renderer = $this->renderer;
              }
              $msgText = $renderer->renderPlain($msgText);
            }
            $this->messenger->addMessage($msgText, $message['status']);
          }
        }
      }
    }
  }

}
