<?php

namespace Drupal\admin_theme_by_content_type\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Theme\ThemeInitializationInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\node\NodeInterface;
use Drupal\node\NodeTypeInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Routing\RouterInterface;

/**
 * Event subscriber to change theme in content types.
 */
class AdminThemeSubscriber implements EventSubscriberInterface {

  /**
   * Event subscriber constructor.
   *
   * @param \Drupal\Core\Theme\ThemeInitializationInterface $themeInitialization
   *   Interface which contain theme initialization logic.
   * @param \Symfony\Component\Routing\RouterInterface $router
   *   The router manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $themeManager
   *   The theme manager.
   */
  public function __construct(
    protected readonly ThemeInitializationInterface $themeInitialization,
    protected readonly RouterInterface $router,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly ThemeManagerInterface $themeManager,
  ) {}

  /**
   * Event handler.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   Event data.
   */
  public function onKernelRequest(RequestEvent $event) {
    $request = $event->getRequest();

    $parameters = [];
    try {
      // The Route Object is an array.  Because.
      $route_object = $this->router->matchRequest($request, $parameters);
    }
    catch (\Exception $e) {
      return;
    }

    if (empty($route_object['_route'])) {
      return;
    }

    $node_bundle = FALSE;

    if ($route_object['_route'] === "entity.node.edit_form") {
      $node = $route_object['node'];
      // @TODO assess whether these safety checks slow down this code
      // measurably, and drop if so, as not having the node (or below, the
      // node type) for these routes would indicate already broken code.
      $node_bundle = $node instanceof NodeInterface ? $node->bundle() : FALSE;
    }
    elseif ($route_object['_route'] === "node.add") {
      $nodeType = $route_object['node_type'];
      $node_bundle = $nodeType instanceof NodeTypeInterface ? $nodeType->id() : FALSE;
    }

    if ($node_bundle) {
      $node_bundles = $this->configFactory->get('admin_theme_by_content_type.settings')->get('node_bundles') ?? [];
      if (in_array($node_bundle, $node_bundles, TRUE)) {
        $system_theme = $this->configFactory->get('system.theme');
        $admin_theme_name = $system_theme->get('admin');
        $this->themeManager->setActiveTheme($this->themeInitialization->initTheme($admin_theme_name));
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onKernelRequest', 35],
    ];
  }

}
