<?php

namespace Drupal\admin_toolbar_content\Plugin\AdminToolbarContent;

use Drupal\admin_toolbar_content\AdminToolbarContentPluginBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Views;

/**
 * An AdminToolbarContentPlugin for altering the system content menu.
 *
 * @see \Drupal\admin_toolbar_content\Plugin\Derivative\AdminToolbarContentMenuLinks.
 *
 * @AdminToolbarContentPlugin(
 *   id = "content",
 *   name = @Translation("Content"),
 *   description = @Translation("Alters the system content menu to provide links for each content type."),
 *   entity_type = "node_type"
 * )
 */
class AdminToolbarContentContentPlugin extends AdminToolbarContentPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    $elements += $this->buildConfigFormRecentItems($form, $form_state);

    $elements['hide_non_content_items'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide non content items'),
      '#description' => $this->t('Hides items under "content" not directly related to content types.'),
      '#default_value' => $this->config->get("plugins." . $this->getPluginId() . '.hide_non_content_items') ?? FALSE,
    ];

    $elements['hide_content_type_items'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Hide content type items'),
      '#description' => $this->t('If one of these conditions is met, the relevant content type item will be hidden.'),
      '#options' => [
        'content_view' => $this->t("If the content type is not available in the 'content' view 'type' filter."),
        'admin_permissions' => $this->t("If the user doesn't have administrative permissions."),
      ],
      '#default_value' => $this->config->get("plugins." . $this->getPluginId() . '.hide_content_type_items') ?? [],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {

    $hide_non_content_items = $this->config->get("plugins." . $this->getPluginId() . '.hide_non_content_items') ?? 0;
    if ($hide_non_content_items) {
      $parents = ['system.admin_content'];
      while (!empty($parents)) {
        $removed = [];
        foreach ($links as $name => $link) {
          if (isset($link['parent']) && in_array($link['parent'], $parents)) {
            if (!str_starts_with($name, 'admin_toolbar_content')) {
              unset($links[$name]);
              $removed[] = $name;
            }
          }
        }
        $parents = $removed;
      }
    }

    // Unset the original "add content" menu item and it's children.
    // These are replaced with the links from createMenuLinkItems.
    unset($links['admin_toolbar_tools.extra_links:node.add']);
    $contentTypes = $this->getItems();
    foreach ($contentTypes as $contentType) {
      unset($links['admin_toolbar_tools.extra_links:node.add.' . $contentType->id()]);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function createMenuLinkItems(): void {
    $this->createCollectionLinks('system.admin_content');
    $this->createItemLinks('system.admin_content', 'type');
    $this->createItemAddLinks('node.add');
    $this->createItemRecentContentLinks('node');
  }

  /**
   * {@inheritdoc}
   */
  protected function createCollectionLink(array $collection, $route_name, $route_parameters = []): void {
    parent::createCollectionLink($collection, $route_name, $route_parameters);
    if (isset($this->links[$collection['id']])) {
      // Because we don't have a custom root item, we add collections to the
      // existing system content menu item.
      if ($collection['parent'] == $this->getPluginId()) {
        $this->links[$collection['id']]['parent'] = 'system.admin_content';
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function createItemLink(mixed $item, string $route_name, string $route_item_parameter): void {
    parent::createItemLink($item, $route_name, $route_item_parameter);
    $collection = $this->getItemCollection($item);
    if (isset($this->links[$collection['id'] . '.' . $item->id()])) {
      // Because we don't have a custom root item, we add items without a
      // collection to the existing system content menu item.
      if ($collection['id'] == $this->getPluginId()) {
        $this->links[$collection['id'] . '.' . $item->id()]['parent'] = 'system.admin_content';
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function filterMenuItem(array &$item, &$parent = NULL): bool {
    // Only fetch this once, so stick it in a static variable.
    static $view_definition = NULL;

    /** @var \Drupal\Core\Url $url */
    $url = $item['url'];
    if ($url->isRouted() && $url->getRouteName() == 'system.admin_content') {
      // Determine if we need to hide content type menu items.
      $params = $url->getRouteParameters();
      $content_type = $params['type'] ?? '';

      if (empty($content_type)) {
        return FALSE;
      }

      $hide_content_type_items = $this->config->get("plugins." . $this->getPluginId() . '.hide_content_type_items') ?? [];

      if (!empty($hide_content_type_items['content_view'])) {
        // If the content type is not set in the content view type filter
        // selection, then don't show the menu item.
        if (!$view_definition) {
          // Check if there is a content type specific view, else fall back to
          // the normal content view of core.
          $view = Views::getView('content_' . $content_type) ?? Views::getView('content');
          if ($view) {
            // Look for the page_1 display (set by core by default). Or fallback
            // to the default settings.
            /** @var \Drupal\views\Plugin\views\ViewsHandlerInterface $handler */
            $view_definition = $view->getHandler('page_1', 'filter', 'type')
              ?? $view->getHandler('default', 'filter', 'type')
              ?? [];
          }
        }

        if (!empty($view_definition['value']) && !in_array($content_type, $view_definition['value'])) {
          return TRUE;
        }
      }

      if (!empty($hide_content_type_items['admin_permissions']) && $this->currentUser->id() > 1) {

        // We need at least permission to see the overview page.
        if (!$this->currentUser->hasPermission('access content overview')) {
          return TRUE;
        }

        // If the user doesn't have at least one of these administrative
        // permissions on the content type, then don't show the menu item.
        $admin_permissions = FALSE;
        $permissions = [
          // General node permissions.
          'bypass node access',
          'administer nodes',
          'view own unpublished content',
          'view all revisions',
          'revert all revisions',
          'delete all revisions',
          // Content type specific permissions.
          "create $content_type content",
          "edit own $content_type content",
          "edit any $content_type content",
          "delete own $content_type content",
          "delete any $content_type content",
          "view $content_type revisions",
          "revert $content_type revisions",
          "delete $content_type revisions",
        ];
        foreach ($permissions as $permission) {
          $admin_permissions |= $this->currentUser->hasPermission($permission);
          if ($admin_permissions) {
            // We know enough, stop checking.
            break;
          }
        }

        if (!$admin_permissions) {
          return TRUE;
        }

      }

    }

    return parent::filterMenuItem($item, $parent);
  }

}
