<?php

namespace Drupal\admin_ui_only;

use Drupal\Core\Routing\RouteBuildEvent;
use Drupal\Core\Routing\RoutingEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Prevents html access to non admin pages.
 */
class EventSubscriber implements EventSubscriberInterface {

  /**
   * Denies access if the format is...
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event to process.
   */
  public function onKernelRequest(RequestEvent $event): void {
    if ($this->deny($event->getRequest())) {
      // @todo make the exception type configurable.
      throw new NotFoundHttpException();
    }
  }

  /**
   * Determines if the request should be denied.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return bool
   *   TRUE if the request should be denied and FALSE if should be allowed.
   */
  private function deny(Request $request): bool {
    if ($request->getRequestFormat() !== 'html') {
      return FALSE;
    }
    /** @var \Symfony\Component\Routing\Route|null $route_object */
    $route_object = $request->attributes->get('_route_object');
    if ($route_object && $route_object->getOption('_admin_route')) {
      // We are not interested in admin routes.
      return FALSE;
    }
    // Give access to front page. The front page should be set to user/login for
    // the best UX.
    return $request->getPathInfo() !== '/';
  }

  /**
   * Alters existing routes for a specific collection.
   *
   * @param \Drupal\Core\Routing\RouteBuildEvent $event
   *   The route build event.
   */
  public function onAlterRoutes(RouteBuildEvent $event): void {
    $collection = $event->getRouteCollection();
    // @todo Allow users to add more routes to this list.
    foreach (['user.login', 'user.logout', 'system.404', 'system.403'] as $route_name) {
      $route = $collection->get($route_name);
      $route?->setOption('_admin_route', TRUE);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // @todo alter user.login and user.logout to use admin route.
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 30];
    $events[RoutingEvents::ALTER][] = ['onAlterRoutes', 30];
    return $events;
  }

}
