<?php

namespace Drupal\advance_currency_converter\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\advance_currency_converter\Controller\CurrencyNameFetch;

/**
 * Class Currency Settings.
 *
 * @category class
 */
class CurrencySettings extends ConfigFormBase implements ContainerInjectionInterface {

  protected $currName = NULL;

  /**
   * Constructor with setting the currency name class object.
   *
   * @param Drupal\advance_currency_converter\Controller\CurrencyNameFetch $currName
   *
   *   It will provide you a Service class Object.
   */
  public function __construct(CurrencyNameFetch $currName) {
    $this->currName = $currName;
  }

  /**
   * It will help us achieve the Dependency Injection.
   *
   * @param Symfony\Component\DependencyInjection\ContainerInterface $container
   *
   *   It will provide you a Service class object.
   *
   * @return static
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('currency.fetch_data')
    );
  }

  /**
   * Get Editable Config name.
   *
   * {@inheritDoc}
   */
  protected function getEditableConfigNames() {
    return ['currency.converter'];
  }

  /**
   * Get form id.
   *
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'currency_converter';
  }

  /**
   * Building a form.
   *
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $options = $this->currName->getInfo();
    $form = parent::buildForm($form, $form_state);
    $form['Currency_Converter_API'] = [
      '#type' => 'details',
      '#title' => $this->t('Currency Converter API'),
      '#open' => TRUE,
    ];
    $form['Currency_Converter_API']['selection'] = [
      '#type' => 'select',
      '#options' => [
        'Select Currency API' => 'Select Currency API',
        'Google Currency Converter API' => 'Google Currency Converter API',
        'Data Offline Handling' => 'Data Offline Handling',
      ],
      '#default_value' => $this->config('currency.converter')->get('selection') ? $this->config('currency.converter')->get('selection') : 'Google Currency Converter API',
      '#weight' => -30,
    ];
    $form['Currency_Converter_API']['graph'] = [
      '#type' => 'checkbox',
      '#title' => 'Do you want to show the graph?' . "<br>" . 'The graph will render'
      . ' when the Data Offline Handling mode selected.',
      '#default_value' => $this->config('currency.converter')->get('graph'),
    ];
    $form['Selection'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select Currency you want to display:'),
      '#options' => $options,
      '#default_value' => $this->config('currency.converter')->get('selecti'),
    ];
    $form['#attached']['library'][] = 'advance_currency_converter/currency-check';
    return $form;
  }

  /**
   * Submitting a form.
   *
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Saving the data into the config table.
    $this->config('currency.converter')->set('graph', $form_state->getValue('graph'))->save();
    $this->config('currency.converter')->set('selection', $form_state->getValue('selection'))->save();
    $this->config('currency.converter')->set('selecti', $form_state->getValue('Selection'))->save();
    parent::submitForm($form, $form_state);
  }

}
