<?php

namespace Drupal\advanced_email_validation\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\advanced_email_validation\EmailValidatorWrapperService;

/**
 * Configure Email Validation settings for this site.
 */
class AdvancedEmailValidationSettingsForm extends ConfigFormBase {

  const ADVANCED_EMAIL_VALIDATION_LISTS = [
    'disposable_list',
    'free_list',
    'banned_list',
  ];

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_email_validation_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['advanced_email_validation.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('advanced_email_validation.settings');

    $form['rules'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Ensure user account email domains are:'),
      '#options' => [
        EmailValidatorWrapperService::ADVANCED_EMAIL_VALIDATION_INVALID => $this->t('Valid (uses an MX lookup)'),
        EmailValidatorWrapperService::ADVANCED_EMAIL_VALIDATION_DISPOSABLE => $this->t('Not a disposable email provider like mailinator.com'),
        EmailValidatorWrapperService::ADVANCED_EMAIL_VALIDATION_FREE => $this->t('Not a public/free email provider like gmail.com'),
        EmailValidatorWrapperService::ADVANCED_EMAIL_VALIDATION_BANNED => $this->t('Not in your custom list of banned email providers'),
      ],
      '#default_value' => $config->get('rules') ?? [],
    ];

    $form['banned_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Banned domains'),
      '#description' => $this->t('Enter one domain per line to the list of banned domains, e.g.:
      <br/>example.org
      <br/>example.com'),
      '#default_value' => implode('/n', $config->get('banned_list') ?? []),
    ];

    $form['lists_help'] = [
      '#markup' => $this->t('Advanced email validation is built using the
        <a href="@library">stymiee/email-validator library</a>, which fetches
        lists of disposable and free domains from the internet, helping it keep
        up to date with the latest developments. If you want to
        <strong>add</strong> more domains to those lists <em>for this site only</em> - e.g. if you discover
        a domain that is not included - you may do so below.',
        ['@library' => 'https://github.com/stymiee/email-validator'])
    ];

    $form['disposable_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional disposable domains'),
      '#description' => $this->t('Enter one domain per line to <strong>add</strong> to the default list of disposable domains, e.g.:
      <br/>example.org
      <br/>example.com'),
      '#default_value' => implode('/n', $config->get('disposable_list') ?? []),
    ];

    $form['free_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional free domains'),
      '#description' => $this->t('Enter one domain per line to <strong>add</strong> to the default list of free domains, e.g.:
      <br/>example.org
      <br/>example.com'),
      '#default_value' => implode('/n', $config->get('free_list') ?? []),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    foreach (self::ADVANCED_EMAIL_VALIDATION_LISTS as $list) {
      $value = $form_state->getValue($list);
      if (preg_match('/@+/', $value)) {
        $form_state->setErrorByName($list, $this->t('Use domain names only (the part after "@" in an email address)'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('advanced_email_validation.settings');

    $config->set('rules', $form_state->getValue('rules'));

    foreach (self::ADVANCED_EMAIL_VALIDATION_LISTS as $list) {
      $values = explode('/n', $form_state->getValue($list));
      foreach ($values as &$value) {
        $value = trim($value);
      }
      $config->set($list, $values);
    }

    $config->save();
    parent::submitForm($form, $form_state);
  }

}
