<?php

namespace Drupal\Tests\advanced_email_validation\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;

/**
 * Tests module configuration.
 *
 * @group advanced_email_validation
 */
class EmailValidationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'advanced_email_validation',
  ];

  /**
   * The advanced email validation service.
   *
   * @var \Drupal\advanced_email_validation\AdvancedEmailValidatorInterface
   */
  protected $emailValidator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig('system');
    $this->installEntitySchema('user');
    $this->installConfig('advanced_email_validation');

    $this->emailValidator = \Drupal::service('advanced_email_validation.validator');
  }

  /**
   * Test basic validation.
   */
  public function testBasicValidation(): void {
    $accountName = $this->randomMachineName();
    $validAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@example.com',
    ]);
    $violations = $validAccount->validate();
    $this->assertEquals(0, $violations->count(), 'Basic validation passed.');

    $invalidEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $this->randomString(),
    ]);
    $violations = $invalidEmailAccount->validate();
    // Core will also fail validation on this one - hence 2 violations.
    $this->assertEquals(2, $violations->count(), 'Basic validation failed.');
    $this->assertEquals('Not a valid email address', $violations[0]->getMessage());

    $newError = $this->randomString();
    $config = $this->config('advanced_email_validation.settings');
    $config->set('error_messages.basic', $newError)->save();
    $violations = $invalidEmailAccount->validate();
    $this->assertEquals($newError, $violations[0]->getMessage());
  }

  /**
   * Test MX Record checking which is turned on by default.
   */
  public function testMxRecordValidation(): void {
    // Test with a user that should pass.
    $accountName = $this->randomMachineName();
    $validMxAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@drupal.org',
    ]);
    $violations = $validMxAccount->validate();
    $this->assertEquals(0, $violations->count(), 'MX record validation passed.');

    // Syntactically correct domain that doesn't exist.
    $invalidMxAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@invalidinvalidinvalid.com',
    ]);
    $violations = $invalidMxAccount->validate();
    $this->assertEquals(1, $violations->count(), 'MX record validation failed.');
    $this->assertEquals('Not a valid email address', $violations[0]->getMessage());

    // Change the error message.
    $newError = $this->randomString();
    $config = $this->config('advanced_email_validation.settings');
    $config->set('error_messages.mx_lookup', $newError)->save();
    $violations = $invalidMxAccount->validate();
    $this->assertEquals($newError, $violations[0]->getMessage());

    // Turn the test off.
    $config->set('rules.mx_lookup', 0)->save();
    $violations = $invalidMxAccount->validate();
    $this->assertEquals(0, $violations->count(), 'MX record validation skipped.');
  }

  /**
   * Test free email provider checking which is turned off by default.
   */
  public function testFreeProviderValidation(): void {
    // Test is skipped.
    $accountName = $this->randomMachineName();
    $freeEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@gmail.com',
    ]);
    $violations = $freeEmailAccount->validate();
    $this->assertEquals(0, $violations->count(), 'Free provider validation skipped.');

    // Turn the test on.
    $config = $this->config('advanced_email_validation.settings');
    $config->set('rules.free', 1)->save();
    $violations = $freeEmailAccount->validate();
    $this->assertEquals(1, $violations->count(), 'Free provider validation failed.');
    $this->assertEquals('Free public email providers are not allowed', $violations[0]->getMessage());

    // Change the error message.
    $newError = $this->randomString();
    $config->set('error_messages.free', $newError)->save();
    $violations = $freeEmailAccount->validate();
    $this->assertEquals($newError, $violations[0]->getMessage());

    // Test with a user that should pass.
    $drupalEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@drupal.org',
    ]);
    $violations = $drupalEmailAccount->validate();
    $this->assertEquals(0, $violations->count(), 'Free provider validation passed.');
  }

  /**
   * Test free email provider list customisation.
   *
   * MUST be a separate test because the list is statically cached by the 3rd
   * party library.
   */
  public function testFreeProviderListCustomisation(): void {
    // User that should pass defaults.
    $accountName = $this->randomMachineName();
    $drupalEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@drupal.org',
    ]);

    // Turn the test on and customise the list.
    $config = $this->config('advanced_email_validation.settings');
    $config->set('rules.free', 1)
      ->set('domain_lists.free', ['drupal.org'])
      ->save();

    // Run validation.
    $violations = $drupalEmailAccount->validate();
    $this->assertEquals(1, $violations->count(), 'Custom free provider validation failed.');
  }

  /**
   * Test disposable email provider checking which is turned off by default.
   */
  public function testDisposableProviderValidation(): void {
    // Test is skipped.
    $accountName = $this->randomMachineName();
    $disposableEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@mailinator.com',
    ]);
    $violations = $disposableEmailAccount->validate();
    $this->assertEquals(0, $violations->count(), 'Disposable provider validation skipped.');

    // Turn the test on.
    $config = $this->config('advanced_email_validation.settings');
    $config->set('rules.disposable', 1)->save();
    $violations = $disposableEmailAccount->validate();
    $this->assertEquals(1, $violations->count(), 'Disposable provider validation failed.');
    $this->assertEquals('Disposable emails are not allowed', $violations[0]->getMessage());

    // Change the error message.
    $newError = $this->randomString();
    $config->set('error_messages.disposable', $newError)->save();
    $violations = $disposableEmailAccount->validate();
    $this->assertEquals($newError, $violations[0]->getMessage());

    // Test with a user that should pass.
    $drupalEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@drupal.org',
    ]);
    $violations = $drupalEmailAccount->validate();
    $this->assertEquals(0, $violations->count(), 'Disposable provider validation passed.');
  }

  /**
   * Test disposable email provider list customisation.
   *
   * MUST be a separate test because the list is statically cached by the 3rd
   * party library.
   */
  public function testDisposableProviderListCustomisation(): void {
    // User that should pass defaults.
    $accountName = $this->randomMachineName();
    $drupalEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@drupal.org',
    ]);

    // Turn the test on and customise the list.
    $config = $this->config('advanced_email_validation.settings');
    $config->set('rules.disposable', 1)
      ->set('domain_lists.disposable', ['drupal.org'])
      ->save();

    // Run validation.
    $violations = $drupalEmailAccount->validate();
    $this->assertEquals(1, $violations->count(), 'Custom disposable provider validation failed.');
  }

  /**
   * Test banned email provider.
   */
  public function testBannedProviderValidation(): void {
    // User that should pass defaults.
    $accountName = $this->randomMachineName();
    $drupalEmailAccount = User::create([
      'name' => $accountName,
      'mail' => $accountName . '@drupal.org',
    ]);

    // Turn the test on and set the list. Don't check for 'test is skipped'
    // because of static caching of lists.
    $config = $this->config('advanced_email_validation.settings');
    $config->set('rules.banned', 1)
      ->set('domain_lists.banned', ['drupal.org'])
      ->save();

    // Run validation.
    $violations = $drupalEmailAccount->validate();
    $this->assertEquals(1, $violations->count(), 'Banned provider validation failed.');
    $this->assertEquals('Emails using this domain are not allowed', $violations[0]->getMessage());

    // Change the error message.
    $newError = $this->randomString();
    $config->set('error_messages.banned', $newError)->save();
    $violations = $drupalEmailAccount->validate();
    $this->assertEquals($newError, $violations[0]->getMessage());
  }

}
