<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenDialogCommand;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\MessageCommand;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormBuilder;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\Config\ConfigFactoryInterface;

/** 
 * Controller for directory creation modal.
 */
class DirectoryModalController extends ControllerBase {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilder
   */
  protected $formBuilder;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;
  
  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new DirectoryModalController.
   *
   * @param \Drupal\Core\Form\FormBuilder $form_builder
   *   The form builder.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(
    FormBuilder $form_builder,
    CsrfTokenGenerator $csrf_token,
    AdvancedFileDestinationManager $destination_manager,
    ConfigFactoryInterface $config_factory
  ) {
    $this->formBuilder = $form_builder;
    $this->csrfToken = $csrf_token;
    $this->destinationManager = $destination_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('form_builder'),
      $container->get('csrf_token'),
      $container->get('advanced_file_destination.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * Get the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Displays the modal form.
   */
  public function modal(Request $request) {
    $response = new AjaxResponse();

    // Get parent directory from form or request
    $parent_directory = $request->query->get('parent_directory');
    if (empty($parent_directory)) {
      $parent_directory = \Drupal::service('advanced_file_destination.manager')->getDestinationDirectory();
    }

    // Build and display the modal form, passing parent directory
    $modal_form = $this->formBuilder->getForm('\Drupal\advanced_file_destination\Form\NewDirectoryModalForm', $parent_directory);
    
    $response->addCommand(new OpenDialogCommand(
      '#afd-directory-dialog',
      $this->t('Create new directory'),
      $modal_form,
      [
        'width' => 'auto',
        'dialogClass' => 'advanced-file-destination-modal',
        'closeOnEscape' => TRUE,
        'modal' => TRUE,
      ]
    ));
    
    return $response;
  }

  /**
   * Creates a new directory via AJAX.
   */
  public function createDirectory(Request $request) {
    $response = new AjaxResponse();
    
    try {
      $directory_name = $request->request->get('directory_name');
      $parent_directory = $request->request->get('parent_directory');
      
      if (empty($directory_name)) {
        throw new \Exception($this->t('Directory name is required.'));
      }

      // Create directory using the manager service with parent directory
      $directory_service = \Drupal::service('advanced_file_destination.manager');
      $new_directory = $directory_service->createNewDirectory($directory_name, $parent_directory);
      
      $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));
      $response->addCommand(new MessageCommand($this->t('Directory created successfully.')));
      
      // Add the directory path to the response for frontend update
      $response->addCommand(new ReplaceCommand(
        '.directory-path-display',
        '<div class="directory-path-display visible">' . $this->t('Created directory: @path', ['@path' => $new_directory]) . '</div>'
      ));
      
      // Update directory selection dropdown
      $directories = $directory_service->getAvailableDirectories();
      $element = [
        '#type' => 'select',
        '#options' => $directories,
        '#value' => $new_directory,
      ];
      $response->addCommand(new ReplaceCommand(
        'select[name="advanced_file_destination[directory]"]',
        \Drupal::service('renderer')->render($element)
      ));
      
    }
    catch (\Exception $e) {
      $response->addCommand(new MessageCommand($e->getMessage(), NULL, ['type' => 'error']));
    }
    
    return $response;
  }

  /**
   * Stores the parent directory in tempstore.
   */
  public function storeParentDirectory(Request $request) {
    $directory = $request->request->get('directory');
    $response = new JsonResponse();
    
    try {
      if (empty($directory)) {
        throw new \Exception('No directory provided');
      }

      // Set in multiple storage locations to ensure persistence
      $directory_service = \Drupal::service('advanced_file_destination.manager');
      
      // Normalize directory path
      if (!preg_match('#^([a-z]+)://#', $directory)) {
        // Get the default directory scheme from settings
        $default_dir = $this->getDefaultDirectory();
        $scheme = 'public://';
        
        // Extract scheme from default directory
        if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
          $scheme = $matches[1] . '://';
        }
        
        $directory = $scheme . ltrim($directory, '/');
      }
      
      // Store in state and session
      \Drupal::state()->set('advanced_file_destination.current_directory', $directory);
      \Drupal::service('session')->set('advanced_file_destination.directory', $directory);
      
      // Store in service
      $result = $directory_service->setDestinationDirectory($directory);
      if (!$result) {
        throw new \Exception('Failed to set directory');
      }

      // Set upload location in static cache for immediate use
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $directory;

      // Log success with actual path
      \Drupal::logger('advanced_file_destination')->notice(
        'Directory explicitly set: @dir', 
        ['@dir' => $directory]
      );

      return new JsonResponse([
        'status' => 'success',
        'directory' => $directory
      ]);
    }
    catch (\Exception $e) {
      \Drupal::logger('advanced_file_destination')->error(
        'Failed to store directory: @error', 
        ['@error' => $e->getMessage()]
      );
      return new JsonResponse([
        'status' => 'error',
        'message' => $e->getMessage()
      ], 500);
    }
  }

  /**
   * Helper function to normalize directory paths.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getDefaultDirectory();
    }
    
    // Remove multiple consecutive slashes
    $path = preg_replace('#/+#', '/', $path);
    // Remove trailing slash
    $path = rtrim($path, '/');
    
    // Get the default directory scheme
    $default_dir = $this->getDefaultDirectory();
    $default_scheme = 'public://';
    
    // Extract scheme from default directory
    if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
      $default_scheme = $matches[1] . '://';
    }
    
    // Handle scheme formatting
    if (!preg_match('#^([a-z]+)://#', $path)) {
      $path = $default_scheme . ltrim($path, '/');
    }
    
    return $path;
  }

  public function submitFormAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    
    try {
      $directory_name = $form_state->getValue('directory_name');
      $parent_directory = $form_state->getValue('parent_directory');
      
      if (empty($directory_name)) {
        throw new \Exception($this->t('Directory name is required.'));
      }

      // Create directory using the manager service with parent directory
      $directory_service = \Drupal::service('advanced_file_destination.manager');
      $new_directory = $directory_service->createNewDirectory($directory_name, $parent_directory);
      
      $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));
      $response->addCommand(new MessageCommand($this->t('Directory created successfully.')));
      
      // Add the directory path to the response for frontend update
      $response->addCommand(new ReplaceCommand(
        '.directory-path-display',
        '<div class="directory-path-display visible">' . $this->t('Created directory: @path', ['@path' => $new_directory]) . '</div>'
      ));
      
      // Update the directory selection dropdown with proper label handling
      $directories = $directory_service->getAvailableDirectories();
      $element = [
        '#type' => 'select',
        '#title' => $this->t('Destination folder'),
        '#title_display' => 'before',
        '#options' => $directories,
        '#value' => $new_directory,
        '#attributes' => [
          'class' => ['directory-selection-field'],
        ],
      ];
      
      // Add command to update the select element
      $response->addCommand(new ReplaceCommand(
        '.directory-selection-group select',
        \Drupal::service('renderer')->render($element)
      ));
      
    }
    catch (\Exception $e) {
      $response->addCommand(new MessageCommand($e->getMessage(), NULL, ['type' => 'error']));
    }
    
    return $response;
  }
}
