<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Component\Utility\Bytes;
use Drupal\Core\Image\ImageFactory;
use Drupal\file\FileInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Form handler for directory add/edit forms.
 */
class AdvancedFileDestinationDirectoryForm extends EntityForm {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The image factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected $imageFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AdvancedFileDestinationDirectoryForm.
   */
  public function __construct(
    FileSystemInterface $file_system,
    MessengerInterface $messenger,
    FileUrlGeneratorInterface $file_url_generator,
    DateFormatterInterface $date_formatter,
    ImageFactory $image_factory,
    ConfigFactoryInterface $config_factory
  ) {
    $this->fileSystem = $file_system;
    $this->messenger = $messenger;
    $this->fileUrlGenerator = $file_url_generator;
    $this->dateFormatter = $date_formatter;
    $this->imageFactory = $image_factory;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('file_system'),
      $container->get('messenger'),
      $container->get('file_url_generator'),
      $container->get('date.formatter'),
      $container->get('image.factory'),
      $container->get('config.factory')
    );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    
    $directory = $this->entity;

    // Set theme hook early and explicitly to ensure it's applied
    $form['#theme'] = 'afd_directory_form';
    
    // Always attach required libraries
    $form['#attached']['library'][] = 'advanced_file_destination/directory_form';
    $form['#attached']['library'][] = 'advanced_file_destination/directory_list';
    $form['#attached']['library'][] = 'advanced_file_destination/directory_selection';

    // Add meta information
    $form['meta'] = [
      '#type' => 'details',
      '#title' => $this->t('Directory information'),
      '#open' => TRUE,
      '#weight' => -10,
      '#attributes' => ['class' => ['entity-meta__header']],
    ];

    if ($directory->getCreatedTime()) {
      $form['meta']['created'] = [
        '#type' => 'item',
        '#title' => $this->t('Created'),
        '#markup' => $this->dateFormatter->format($directory->getCreatedTime(), 'medium'),
        '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
        '#weight' => 0,
      ];
    }

    if ($directory->getChangedTime() && !$directory->isNew()) {
      $form['meta']['changed'] = [
        '#type' => 'item',
        '#title' => $this->t('Last saved'),
        '#markup' => $this->dateFormatter->format($directory->getChangedTime(), 'medium'),
        '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
        '#weight' => 1,
      ];
    }

    // Add revision information group
    $form['revision_information'] = [
      '#type' => 'details',
      '#title' => $this->t('Revision information'),
      '#weight' => 20,
      '#open' => TRUE,
      '#attributes' => ['class' => ['entity-meta__revision']],
    ];

    $form['revision_information']['revision'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create new revision'),
      '#default_value' => $directory->isNew() || $directory->isNewRevision(),
      '#weight' => -1,
    ];

    $form['revision_information']['revision_log'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Revision log message'),
      '#description' => $this->t('Briefly describe the changes you have made.'),
      '#default_value' => $directory->getRevisionLogMessage(),
      '#rows' => 4,
      '#states' => [
        'visible' => [
          ':input[name="revision"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#maxlength' => 255,
      '#default_value' => $directory->label(),
      '#description' => $this->t('The human-readable name of the directory.'),
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => !$directory->isNew() ? $directory->status() : TRUE,
      '#description' => $this->t('Whether the directory is active and available for use.'),
    ];

    $form['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Directory path'),
      '#default_value' => $directory->getPath(),
      '#description' => $this->t('The directory path (e.g., @default_dir/custom).', [
        '@default_dir' => $this->getDefaultDirectory(),
      ]),
      '#required' => TRUE,
    ];

    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $directory->getWeight(),
      '#description' => $this->t('Optional. Directories are displayed in ascending order by weight.'),
    ];

    // Get all user roles except anonymous and authenticated
    $roles = \Drupal::entityTypeManager()->getStorage('user_role')->loadMultiple();
    unset($roles['anonymous']);
    unset($roles['authenticated']);
    
    $role_options = [];
    foreach ($roles as $role_id => $role) {
      $role_options[$role_id] = $role->label();
    }
    
    if (!empty($role_options)) {
      $form['roles'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Restrict by role'),
        '#description' => $this->t('Select which roles can use this directory. Leave empty to allow all roles.'),
        '#options' => $role_options,
        '#default_value' => $directory->getRoles(),
      ];
    }

    $form['files'] = [
      '#type' => 'details',
      '#title' => $this->t('Files in directory'),
      '#open' => TRUE,
      '#weight' => 100,
    ];

    // Get files from directory
    $path = $directory->getPath();
    if ($path && file_exists($path)) {
      $files = scandir($path);
      $rows = [];

      foreach ($files as $filename) {
        if ($filename === '.' || $filename === '..') {
          continue;
        }

        $full_path = $path . '/' . $filename;
        $is_dir = is_dir($full_path);

        // Build preview with proper icon styling
        $preview = [
          '#type' => 'container',
          '#attributes' => ['class' => ['directory-preview-wrapper']],
          'icon' => [
            '#type' => 'html_tag',
            '#tag' => 'span',
            '#attributes' => [
              'class' => [$is_dir ? 'directory-icon' : 'file-icon'],
              'title' => $is_dir ? $this->t('Directory') : $this->t('File'),
            ],
          ],
        ];

        if (!$is_dir && $this->isImage($full_path)) {
          $preview = [
            '#theme' => 'image_style',
            '#style_name' => 'thumbnail',
            '#uri' => $full_path,
          ];
        }

        // Convert file path to proper URI if needed
        $default_dir = $this->getDefaultDirectory();
        // Get scheme from default directory
        $scheme = 'public://';
        if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
          $scheme = $matches[1] . '://';
        }
        
        if (strpos($full_path, $scheme) === false && strpos($full_path, 'private://') === false) {
          // Convert /sites/default/files/ paths to proper URIs
          $full_path = str_replace('/sites/default/files/', $scheme, $full_path);
          if (strpos($full_path, $scheme) !== 0 && strpos($full_path, 'private://') !== 0) {
            $full_path = $scheme . ltrim($full_path, '/');
          }
        }

        // Generate proper URL
        try {
          $absolute_url = $this->fileUrlGenerator->generateAbsoluteString($full_path);
          $url = Url::fromUri($absolute_url);
        }
        catch (\Exception $e) {
          \Drupal::logger('advanced_file_destination')->error(
            'Error generating URL for @uri: @error',
            ['@uri' => $full_path, '@error' => $e->getMessage()]
          );
          continue;
        }

        // Build operations
        $operations = [
          'view' => [
            'title' => $this->t('View'),
            'url' => $url,
          ],
        ];

        // Add edit operation if it's a directory
        if (is_dir($full_path)) {
          $operations['edit'] = [
            'title' => $this->t('Edit'),
            'url' => Url::fromRoute('entity.afd_directory.edit_form', ['afd_directory' => $filename]),
          ];
        }
        // Add edit operation for media files
        else {
          $file_entity = \Drupal::entityTypeManager()
            ->getStorage('file')
            ->loadByProperties(['uri' => $full_path]);
          
          if ($file_entity = reset($file_entity)) {
            $media_items = \Drupal::entityTypeManager()
              ->getStorage('media')
              ->loadByProperties(['thumbnail' => $file_entity->id()]);
            
            if ($media_item = reset($media_items)) {
              $operations['edit'] = [
                'title' => $this->t('Edit'),
                'url' => $media_item->toUrl('edit-form'),
              ];
            }
          }
        }

        $rows[] = [
          'preview' => [
            'data' => $preview,
          ],
          'name' => [
            'data' => [
              '#type' => 'link',
              '#title' => $filename,
              '#url' => $url,
            ],
          ],
          'modified' => $this->dateFormatter->format(filemtime($full_path), 'short'),
          'operations' => [
            'data' => [
              '#type' => 'operations',
              '#links' => $operations,
            ],
          ],
        ];
      }

      $form['files'] = [
        '#type' => 'table',
        '#header' => [
          'preview' => $this->t('Preview'),
          'name' => $this->t('Name'),
          'modified' => $this->t('Modified'),
          'operations' => $this->t('Operations'),
        ],
        '#rows' => $rows,
        '#empty' => $this->t('No files found'),
      ];
    }
    else {
      $form['files']['error'] = [
        '#markup' => $this->t('Directory does not exist.'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $path = $form_state->getValue('path');
    
    // Ensure the path has a valid scheme
    if (strpos($path, '://') === FALSE) {
      $default_dir = $this->getDefaultDirectory();
      // Extract scheme from default directory
      $scheme = 'public://';
      if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
        $scheme = $matches[1] . '://';
      }
      
      $path = $scheme . $path;
      $form_state->setValue('path', $path);
    }

    // Try to create/verify the directory
    try {
      if (!$this->fileSystem->prepareDirectory($path, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        $form_state->setErrorByName('path', $this->t('The directory could not be created or is not writable.'));
      }
    }
    catch (\Exception $e) {
      $form_state->setErrorByName('path', $this->t('Error checking directory: @error', ['@error' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $directory = $this->entity;
    
    // Set the values from the form to the entity
    $values = $form_state->getValues();
    
    // Handle revision settings - check if the entity type supports revisions first
    if ($directory->getEntityType()->hasKey('revision') && isset($values['revision'])) {
      $directory->setNewRevision($values['revision']);
      if ($values['revision'] && isset($values['revision_log'])) {
        $directory->setRevisionLogMessage($values['revision_log']);
        $directory->setRevisionUserId(\Drupal::currentUser()->id());
      }
    }
    
    // Process roles (filter out unchecked roles)
    if (isset($values['roles']) && is_array($values['roles'])) {
      $roles = array_filter($values['roles'], function($item) {
        return !empty($item);
      });
      $directory->setRoles(array_keys($roles));
    }
    
    // Ensure status is properly set
    $directory->setStatus($values['status']);
    
    // Save the entity
    $status = $directory->save();

    // Set success message and redirect
    $message = $status === SAVED_NEW 
      ? $this->t('Created directory %label.', ['%label' => $directory->label()])
      : $this->t('Updated directory %label.', ['%label' => $directory->label()]);
    
    $this->messenger->addStatus($message);
    
    // Redirect to the directory list
    $form_state->setRedirect('entity.afd_directory.collection');
  }

  /**
   * Determines if a file is an image.
   *
   * @param string $uri
   *   The URI of the file.
   *
   * @return bool
   *   TRUE if the file is an image, FALSE otherwise.
   */
  protected function isImage($uri) {
    $image = $this->imageFactory->get($uri);
    return $image->isValid();
  }

  /**
   * Helper function to get file entity by URI.
   */
  protected function getFileEntity($uri) {
    try {
      $files = \Drupal::entityTypeManager()
        ->getStorage('file')
        ->loadByProperties(['uri' => $uri]);
      return reset($files);
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Helper function to get media entities by file entity.
   */
  protected function getMediaEntities($file_entity) {
    try {
      return \Drupal::entityTypeManager()
        ->getStorage('media')
        ->loadByProperties(['thumbnail' => $file_entity->id()]);
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    
    // Ensure the submit button has the right properties
    $actions['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#submit' => ['::submitForm', '::save'],
      '#button_type' => 'primary',
      '#weight' => 5,
    ];
    
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Call parent submitForm which builds the entity from form values
    parent::submitForm($form, $form_state);
  }

  // The existing save() method will be automatically called after submitForm

  /**
   * {@inheritdoc}
   */
  protected function copyFormValuesToEntity(\Drupal\Core\Entity\EntityInterface $entity, array $form, FormStateInterface $form_state) {
    // Get all values from form state
    $values = $form_state->getValues();
    
    // Handle revision value separately - extract and remove it before parent method processes values
    if (isset($values['revision'])) {
      $create_revision = $values['revision'];
      unset($values['revision']);
      
      // Handle revision flag separately using proper methods
      if ($entity->getEntityType()->hasKey('revision')) {
        // Check if entity supports the revision methods before calling them
        if (method_exists($entity, 'setNewRevision')) {
          $entity->setNewRevision($create_revision);
          
          if ($create_revision && isset($values['revision_log'])) {
            if (method_exists($entity, 'setRevisionLogMessage')) {
              $entity->setRevisionLogMessage($values['revision_log']);
            }
            
            if (method_exists($entity, 'setRevisionUserId')) {
              $entity->setRevisionUserId(\Drupal::currentUser()->id());
            }
          }
        }
      }
    }
    
    // Filter values to only include actual field names
    $field_names = array_keys($entity->getFieldDefinitions());
    $filtered_values = array_intersect_key($values, array_flip($field_names));
    
    // Let parent handle the entity fields that actually exist
    foreach ($filtered_values as $key => $value) {
      $entity->set($key, $value);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Debug template
    $form['#theme'] = 'afd_directory_form';
    // Return the parent form
    return parent::buildForm($form, $form_state);
  }

}
