<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;

/**
 * Configure Advanced File Destination settings.
 */
class AdvancedFileDestinationSettingsForm extends ConfigFormBase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * Constructs a new AdvancedFileDestinationSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed configuration manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, FileSystemInterface $file_system, StreamWrapperManagerInterface $stream_wrapper_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->fileSystem = $file_system;
    $this->streamWrapperManager = $stream_wrapper_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('file_system'),
      $container->get('stream_wrapper_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['advanced_file_destination.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('advanced_file_destination.settings');
    
    // Get stored values with fallbacks
    $current_default_directory = $config->get('default_directory') ?: 'public://';
    $saved_stream_wrapper = $config->get('stream_wrapper');
    $saved_subdirectory = $config->get('subdirectory') ?: '';
    
    // Split the default directory into scheme and subdirectory if needed
    $scheme = $saved_stream_wrapper ?: 'public://';
    $subdirectory = $saved_subdirectory;
    
    // If we don't have explicitly saved stream wrapper/subdirectory,
    // try to extract them from the default directory
    if (empty($saved_stream_wrapper) && preg_match('#^([a-z]+://)(.*)#', $current_default_directory, $matches)) {
      $scheme = $matches[1];
      // Only use subdirectory from default_directory if it wasn't explicitly saved
      if (empty($saved_subdirectory)) {
        $subdirectory = $matches[2];
      }
    }

    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => TRUE,
    ];
    
    // Add stream wrapper visibility settings before loading the wrappers
    $form['general']['stream_visibility'] = [
      '#type' => 'details',
      '#title' => $this->t('Stream wrapper visibility'),
      '#open' => FALSE,
      '#description' => $this->t('Control which stream wrappers are available in the destination selector.'),
    ];
    $form['general']['stream_visibility']['include_public'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include public:// stream'),
      '#description' => $this->t('Show the public file system in the destination options.'),
      '#default_value' => $config->get('include_public') ?? TRUE,
      '#ajax' => [
        'callback' => '::updateStreamOptions',
        'wrapper' => 'stream-wrapper-container',
        'event' => 'change',
      ],
    ];    
    $form['general']['stream_visibility']['include_private'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include private:// stream (Premium)'),
      '#description' => $this->t('Show the private file system in the destination options.'),
      '#default_value' => $config->get('include_private') ?? FALSE,
      '#ajax' => [
        'callback' => '::updateStreamOptions',
        'wrapper' => 'stream-wrapper-container',
        'event' => 'change',
      ],
      '#disabled' => TRUE,
    ];
    
    $form['general']['stream_visibility']['include_assets'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include assets:// stream (Premium)'),
      '#description' => $this->t('Show the assets file system in the destination options.'),
      '#default_value' => $config->get('include_assets') ?? FALSE,
      '#ajax' => [
        'callback' => '::updateStreamOptions',
        'wrapper' => 'stream-wrapper-container',
        'event' => 'change',
      ],
      '#disabled' => TRUE,
    ];
    
    $form['general']['stream_visibility']['include_temporary'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include temporary:// stream (Premium)'),
      '#description' => $this->t('Show the temporary file system in the destination options.'),
      '#default_value' => $config->get('include_temporary') ?? FALSE,
      '#ajax' => [
        'callback' => '::updateStreamOptions',
        'wrapper' => 'stream-wrapper-container',
        'event' => 'change',
      ],
      '#disabled' => TRUE,
    ];
    
    // Create a container for the directory components
    $form['general']['directory_settings'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['directory-settings-container']],
    ];

    // Load stream wrappers based on current form state or config
    $stream_options = $this->getStreamOptions($form_state);

    $form['general']['directory_settings']['stream_wrapper'] = [
      '#type' => 'select',
      '#title' => $this->t('Default file system'),
      '#options' => $stream_options,
      '#default_value' => $scheme,
      '#required' => TRUE,
      '#description' => $this->t('Select the default file system for uploads.'),
      '#prefix' => '<div id="stream-wrapper-container">',
      '#suffix' => '</div>',
    ];

    $form['general']['directory_settings']['subdirectory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subdirectory (Premium)'),
      '#default_value' => $subdirectory,
      '#description' => $this->t('Optional subdirectory within the selected file system (e.g., "custom_uploads"). Do not include leading or trailing slashes.'),
      '#disabled' => TRUE,
    ];
    
    // Replace display element with an element that has a specific ID for JS targeting
    $form['general']['directory_settings']['display_path'] = [
      '#type' => 'item',
      '#title' => $this->t('Full directory path'),
      '#markup' => '<div id="default-directory-wrapper"><code id="default-directory-preview">' . $current_default_directory . '</code></div>',
      '#description' => $this->t('This is the full path where files will be stored.'),
    ];
    
    // Add hidden field to store the complete path (needed for form submission)
    $form['general']['default_directory'] = [
      '#type' => 'hidden',
      '#value' => $current_default_directory,
      '#attributes' => [
        'id' => 'afd-default-directory',
      ],
    ];
    
    // Attach our custom JavaScript for real-time preview updates
    $form['#attached']['library'][] = 'advanced_file_destination/directory_settings';

    $form['general']['enabled_entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enable for entity types'),
      '#description' => $this->t('Select which entity types should have the advanced file destination functionality.'),
      '#options' => [
        'media' => $this->t('Media'),
        'node' => $this->t('Content'),
        'file' => $this->t('File (Premium)'),
        'user' => $this->t('User (Premium)'),
        'taxonomy_term' => $this->t('Taxonomy term (Premium)'),
        'paragraph' => $this->t('Paragraph (Premium)'),
      ],
      '#default_value' => $config->get('enabled_entity_types') ?: [],
    ];
    $form['general']['enabled_entity_types']['file']['#disabled'] = TRUE;
    $form['general']['enabled_entity_types']['user']['#disabled'] = TRUE;
    $form['general']['enabled_entity_types']['taxonomy_term']['#disabled'] = TRUE;
    $form['general']['enabled_entity_types']['paragraph']['#disabled'] = TRUE;
    
    $form['general']['scan_filesystem'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Scan filesystem for directories (Premium)'),
      '#description' => $this->t('If enabled, the module will scan the filesystem for existing directories in addition to the configured ones.'),
      '#default_value' => $config->get('scan_filesystem') ?? FALSE,
      '#disabled' => TRUE,
    ];

    // Add support for commercial features if premium version.
    $form['premium'] = [
      '#type' => 'details',
      '#title' => $this->t('Premium Features'),
      '#open' => TRUE,
      '#description' => $this->t('These features are available in the premium version of the module.'),
    ];

    $form['premium']['license_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('License key'),
      '#description' => $this->t('Enter your premium license key to unlock additional features.'),
      '#default_value' => $config->get('license_key') ?: '',
    ];

    $form['premium']['features'] = [
      '#type' => 'item',
      '#markup' => '<div class="premium-features-list">' .
        '<h4>' . $this->t('Premium features include:') . '</h4>' .
        '<ul>' .
        '<li>' . $this->t('Access control per directory') . '</li>' .
        '<li>' . $this->t('Automatic file organization rules') . '</li>' .
        '<li>' . $this->t('Advanced file renaming patterns') . '</li>' .
        '<li>' . $this->t('Media library integration') . '</li>' .
        '<li>' . $this->t('Content type-specific folder settings') . '</li>' .
        '</ul></div>',
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];
 
    $form['advanced']['widget_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Widget position'),
      '#description' => $this->t('Where to display the directory selection widget relative to the file upload field.'),
      '#options' => [
        'before' => $this->t('Before upload field'),
        'after' => $this->t('After upload field'),
        'inline' => $this->t('Inline with upload field'),
      ],
      '#default_value' => $config->get('widget_position') ?: 'before',
    ];

    $form['advanced']['display_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Display style'),
      '#description' => $this->t('How to display the directory selection interface.'),
      '#options' => [
        'select' => $this->t('Dropdown select'),
        'radios' => $this->t('Radio buttons'),
        'tree' => $this->t('Tree view (Premium)'),
      ],
      '#default_value' => $config->get('display_style') ?: 'select',
    ];

    // Add JavaScript to update the hidden field value when form elements change
    $form['#attached']['drupalSettings']['advancedFileDestination'] = [
      'currentDirectory' => $current_default_directory,
    ];
    
    return parent::buildForm($form, $form_state);
  }

  /**
   * Check if a stream wrapper is writable.
   *
   * @param string $scheme
   *   The stream wrapper scheme to check.
   *
   * @return bool
   *   TRUE if the stream wrapper is writable, FALSE otherwise.
   */
  protected function isWritableStreamWrapper($scheme) {
    try {
      // Get the stream wrapper instance
      $wrapper = $this->streamWrapperManager->getViaScheme($scheme);
      if ($wrapper) {
        // Check if the wrapper exists and is writable
        return method_exists($wrapper, 'getDirectoryPath') && is_writable($wrapper->getDirectoryPath());
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('advanced_file_destination')->notice('Stream wrapper check error: @error', ['@error' => $e->getMessage()]);
    }
    
    return FALSE;
  }
  
  /**
   * Get the display name for a stream wrapper.
   *
   * @param string $scheme
   *   The stream wrapper scheme.
   * @param array $wrapper
   *   The wrapper info array.
   *
   * @return string
   *   A display name for the stream wrapper.
   */
  protected function getStreamWrapperName($scheme, array $wrapper) {
    // Try to get the name from the wrapper info
    $name = !empty($wrapper['name']) ? $wrapper['name'] : $scheme;
    
    return $this->t('@name (@scheme://)', [
      '@name' => $name,
      '@scheme' => $scheme,
    ]);
  }

  /**
   * Get stream options based on current form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state, if available.
   * 
   * @return array
   *   An array of stream wrapper options.
   */
  protected function getStreamOptions(FormStateInterface $form_state = NULL) {
    $stream_options = [];
    
    // Determine if we should include each stream type
    $config = $this->config('advanced_file_destination.settings');
    
    if ($form_state && !$form_state->isProcessingInput()) {
      // Form is being built initially, use config values
      $include_public = $config->get('include_public') ?? TRUE;
      $include_private = $config->get('include_private') ?? FALSE;
      $include_assets = $config->get('include_assets') ?? FALSE;
      $include_temporary = $config->get('include_temporary') ?? FALSE;
    }
    elseif ($form_state) {
      // Form is being rebuilt via AJAX, use form state values
      $include_public = (bool) $form_state->getValue('include_public');
      $include_private = (bool) $form_state->getValue('include_private');
      $include_assets = (bool) $form_state->getValue('include_assets');
      $include_temporary = (bool) $form_state->getValue('include_temporary');
    }
    else {
      // Fallback to config values
      $include_public = $config->get('include_public') ?? TRUE;
      $include_private = $config->get('include_private') ?? FALSE;
      $include_assets = $config->get('include_assets') ?? FALSE;
      $include_temporary = $config->get('include_temporary') ?? FALSE;
    }
    
    // Force convert to boolean to ensure correct comparison
    $include_public = (bool) $include_public;
    $include_private = (bool) $include_private;
    $include_assets = (bool) $include_assets;
    $include_temporary = (bool) $include_temporary;
    
    try {
      // Get all stream wrappers
      $wrappers = $this->streamWrapperManager->getWrappers();
      
      // For debugging
      \Drupal::logger('advanced_file_destination')->notice('Stream wrapper settings: public=@public, private=@private, assets=@assets, temporary=@temporary', [
        '@public' => $include_public ? 'enabled' : 'disabled',
        '@private' => $include_private ? 'enabled' : 'disabled',
        '@assets' => $include_assets ? 'enabled' : 'disabled',
        '@temporary' => $include_temporary ? 'enabled' : 'disabled',
      ]);
      
      // Filter for writable stream wrappers
      foreach ($wrappers as $scheme_key => $wrapper) {
        // Debug which wrappers we found
        \Drupal::logger('advanced_file_destination')->notice('Found stream wrapper: @scheme', ['@scheme' => $scheme_key]);
        
        // Skip specific schemes based on configuration
        if (($scheme_key === 'public' && !$include_public) ||
            ($scheme_key === 'private' && !$include_private) || 
            ($scheme_key === 'assets' && !$include_assets) || 
            ($scheme_key === 'temporary' && !$include_temporary)) {
          \Drupal::logger('advanced_file_destination')->notice('Skipping stream wrapper: @scheme based on settings', ['@scheme' => $scheme_key]);
          continue;
        }
        
        // Check if the wrapper is writable
        if ($this->isWritableStreamWrapper($scheme_key)) {
          // Get readable name for the stream wrapper
          $stream_name = $this->getStreamWrapperName($scheme_key, $wrapper);
          $stream_options[$scheme_key . '://'] = $stream_name;
          \Drupal::logger('advanced_file_destination')->notice('Added stream wrapper to options: @scheme', ['@scheme' => $scheme_key]);
        }
        else {
          \Drupal::logger('advanced_file_destination')->notice('Stream wrapper not writable: @scheme', ['@scheme' => $scheme_key]);
        }
      }
      
      // If no stream wrappers found, provide at least public:// as a fallback
      if (empty($stream_options)) {
        \Drupal::logger('advanced_file_destination')->notice('No stream wrappers found, adding public:// as fallback');
        $stream_options['public://'] = $this->t('Public files (public://)');
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('advanced_file_destination')->error('Error retrieving stream wrappers: @error', ['@error' => $e->getMessage()]);
      // Ensure we at least have public:// as a fallback
      $stream_options['public://'] = $this->t('Public files (public://)');
    }
    
    return $stream_options;
  }
  
  /**
   * Ajax callback to update the stream wrapper options.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The updated form element.
   */
  public function updateStreamOptions(array &$form, FormStateInterface $form_state) {
    // Just return the stream wrapper element which will be replaced
    return $form['general']['directory_settings']['stream_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Combine the stream wrapper and subdirectory to create the full path
    $stream_wrapper = $form_state->getValue('stream_wrapper');
    $subdirectory = trim($form_state->getValue('subdirectory'));
    
    // Build the complete directory path
    $directory = $stream_wrapper;
    if (!empty($subdirectory)) {
      // Ensure no leading/trailing slashes in subdirectory
      $subdirectory = trim($subdirectory, '/');
      $directory .= $subdirectory;
    }
    
    // Set the combined value for storage
    $form_state->setValue('default_directory', $directory);
    
    // Validate that the directory exists or can be created
    if (!file_exists($directory)) {
      try {
        if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY)) {
          $form_state->setErrorByName('directory_settings', $this->t('The directory %directory could not be created or is not writable.', ['%directory' => $directory]));
        }
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('directory_settings', $this->t('Error creating directory: @error', ['@error' => $e->getMessage()]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get values for storage
    $stream_wrapper = $form_state->getValue('stream_wrapper');
    $subdirectory = trim($form_state->getValue('subdirectory'));
    
    // Build the complete directory path (same logic as in validateForm)
    $default_directory = $stream_wrapper;
    if (!empty($subdirectory)) {
      // Ensure no leading/trailing slashes in subdirectory
      $subdirectory = trim($subdirectory, '/');
      $default_directory .= $subdirectory;
    }
    
    // Save all configuration values
    $this->config('advanced_file_destination.settings')
      ->set('default_directory', $default_directory)
      ->set('stream_wrapper', $stream_wrapper)
      ->set('subdirectory', $subdirectory)
      ->set('scan_filesystem', (bool) $form_state->getValue('scan_filesystem'))
      ->set('license_key', $form_state->getValue('license_key'))
      ->set('widget_position', $form_state->getValue('widget_position'))
      ->set('display_style', $form_state->getValue('display_style'))
      ->set('enabled_entity_types', array_filter($form_state->getValue('enabled_entity_types')))
      ->set('include_public', (bool) $form_state->getValue('include_public'))
      ->set('include_private', (bool) $form_state->getValue('include_private'))
      ->set('include_assets', (bool) $form_state->getValue('include_assets'))
      ->set('include_temporary', (bool) $form_state->getValue('include_temporary'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
