<?php

namespace Drupal\advanced_file_destination\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Plugin\Field\FieldWidget\FileWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Component\Utility\Html;

/**
 * Plugin implementation of the 'advanced_file_destination' widget.
 *
 * @FieldWidget(
 *   id = "advanced_file_destination",
 *   label = @Translation("File with destination selection"),
 *   field_types = {
 *     "file",
 *     "image"
 *   }
 * )
 */
class AdvancedFileDestinationWidget extends FileWidget {

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * Constructs an AdvancedFileDestinationWidget object.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, AdvancedFileDestinationManager $destination_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->destinationManager = $destination_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('advanced_file_destination.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    $selected_directory = $this->destinationManager->getDestinationDirectory();
    if ($selected_directory) {
      // Set upload location for the main element
      $element['#upload_location'] = $selected_directory;
      $element['#upload_validators']['file_directory'] = ['directory' => $selected_directory];
      
      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $selected_directory;
        $element['upload']['#upload_validators']['file_directory'] = ['directory' => $selected_directory];
      }

      // Store in form state and static cache
      $form_state->set(['file_destination', $items->getName(), $delta], $selected_directory);
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $selected_directory;
    }

    // Generate a unique wrapper ID for this element
    $wrapper_id = Html::getUniqueId('advanced-file-destination-wrapper-' . $this->fieldDefinition->getName() . '-' . $delta);
    
    // Get directory service
    $directory_service = \Drupal::service('advanced_file_destination.manager');
    
    // Get current directory - check multiple sources in order
    $current_directory = NULL;
    
    // 1. Check form state first
    $current_directory = $form_state->get(['file_destination', $this->fieldDefinition->getName(), $delta]);
    
    // 2. Check service if not in form state
    if (!$current_directory) {
      $current_directory = $directory_service->getDestinationDirectory();
    }
    
    // 3. Fallback to field settings if still not found
    if (!$current_directory) {
      $field_settings = $this->fieldDefinition->getSettings();
      if (!empty($field_settings['file_directory'])) {
        $scheme = !empty($field_settings['uri_scheme']) ? $field_settings['uri_scheme'] : 'public';
        $current_directory = $scheme . '://' . trim($field_settings['file_directory'], '/');
      }
    }

    // Add directory selection element
    $element['directory'] = [
      '#type' => 'select',
      '#title' => $this->t('Destination folder'),
      '#options' => $directory_service->getAvailableDirectoriesWithTemporary($current_directory),
      '#default_value' => $current_directory,
      '#weight' => -10,
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
      '#ajax' => [
        'callback' => [$this, 'updateDirectoryCallback'],
        'wrapper' => $wrapper_id,
        'event' => 'change',
        'progress' => ['type' => 'throbber'],
      ],
      '#element_validate' => [[$this, 'validateDirectory']],
    ];

    // Set upload location immediately
    if ($current_directory) {
      $element['#upload_location'] = $current_directory;
      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $current_directory;
      }
      
      // Store in form state for persistence
      $form_state->set(['file_destination', $this->fieldDefinition->getName(), $delta], $current_directory);
      
      // Store in manager service
      $directory_service->setDestinationDirectory($current_directory);
      
      // Set in static cache
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $current_directory;
    }

    // Get directory service
    $directory_service = \Drupal::service('advanced_file_destination.manager');
    $selected_directory = $directory_service->getDestinationDirectory();

    if ($selected_directory) {
      // Force upload location in all possible places
      $element['#upload_location'] = $selected_directory;
      $element['#file_directory'] = $selected_directory;

      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $selected_directory;
        $element['upload']['#file_directory'] = $selected_directory;
      }

      // Set upload validators
      if (!isset($element['#upload_validators'])) {
        $element['#upload_validators'] = [];
      }
      $element['#upload_validators']['file_directory'] = ['directory' => $selected_directory];

      // Store in form state
      $form_state->set(['file_destination', $items->getName(), $delta], $selected_directory);
    }

    return $element;
  }

  /**
   * Ajax callback for directory selection.
   */
  public function updateDirectoryCallback(array $form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $element = NestedArray::getValue($form, array_slice($trigger['#array_parents'], 0, -1));
    
    // Get the selected directory
    $directory = $trigger['#value'];
    
    if ($directory) {
      // Update service
      \Drupal::service('advanced_file_destination.manager')->setDestinationDirectory($directory);
      
      // Store in form state
      $form_state->set(
        ['file_destination', $this->fieldDefinition->getName(), $element['#delta']], 
        $directory
      );
      
      // Update upload location
      $element['#upload_location'] = $directory;
      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $directory;
      }
    }
    
    return $element;
  }

  /**
   * Validates the selected directory.
   */
  public function validateDirectory($element, FormStateInterface $form_state, $form) {
    $directory = $element['#value'];
    if ($directory) {
      $file_system = \Drupal::service('file_system');
      if (!$file_system->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        $form_state->setError($element, $this->t('Selected directory is not writable.'));
      }
    }
  }

  /**
   * Get the current directory for this field item.
   */
  protected function getCurrentDirectory(FieldItemListInterface $items, $delta) {
    // For existing files, get their current directory
    if (!$items->isEmpty() && isset($items[$delta]->entity)) {
      $file = $items[$delta]->entity;
      if ($file) {
        $uri = $file->getFileUri();
        if ($uri) {
          return dirname($uri);
        }
      }
    }

    // Get directory from widget state
    $form_state = $items->getEntity()->form_state;
    if ($form_state) {
      $widget_state = static::getWidgetState($form_state->getCompleteForm()['#parents'], $this->fieldDefinition->getName(), $form_state);
      if (isset($widget_state[$delta]['directory'])) {
        return $widget_state[$delta]['directory'];
      }
    }

    // Fallback to default directory
    return $this->destinationManager->ensureDefaultDirectory();
  }

  /**
   * Validate callback for directory selection.
   */
  public static function validateDirectorySelection($element, FormStateInterface $form_state, $form) {
    $file_system = \Drupal::service('file_system');
    $selected_directory = $element['directory']['#value'];
    
    if ($selected_directory) {
      if (!$file_system->prepareDirectory($selected_directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        $form_state->setError($element['directory'], t('Selected directory is not writable.'));
      }
    }
  }

  /**
   * Submit callback for directory selection.
   */
  public static function submitDirectorySelection($element, FormStateInterface $form_state) {
    $selected_directory = $element['directory']['#value'];
    if ($selected_directory) {
      // Store in form state for file processing
      $field_name = $element['#field_name'];
      $delta = $element['#delta'];
      $form_state->set(['file_destination', $field_name, $delta], $selected_directory);
      
      // Set in service for immediate use
      \Drupal::service('advanced_file_destination.manager')
        ->setDestinationDirectory($selected_directory);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function extractFormValues(FieldItemListInterface $items, array $form, FormStateInterface $form_state) {
    $field_name = $this->fieldDefinition->getName();
    $widget_state = static::getWidgetState($form['#parents'], $field_name, $form_state);
    
    // Store directory changes for each delta
    foreach ($items as $delta => $item) {
      if (isset($widget_state[$delta]['directory']) && isset($item->entity)) {
        $form_state->set([
          'advanced_file_destination',
          $field_name,
          $delta,
        ], [
          'file_id' => $item->entity->id(),
          'new_directory' => $widget_state[$delta]['directory'],
        ]);
      }
    }
    
    return parent::extractFormValues($items, $form, $form_state);
  }
}
