<?php

namespace Drupal\Tests\advanced_file_destination\FunctionalJavascript;

use Drupal\Core\File\FileSystemInterface;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\file\Entity\File;

/**
 * Tests the Advanced File Destination form functionality.
 *
 * @group advanced_file_destination
 */
class AdvancedFileDestinationFormTest extends WebDriverTestBase {
  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'node',
    'file',
    'field',
    'field_ui',
    'media',
    'media_library',
    'advanced_file_destination',
    'config',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    // Ignore schema errors during testing
    $this->strictConfigSchema = FALSE;
    parent::setUp();

    // Create test content type
    $this->drupalCreateContentType(['type' => 'article']);

    // Create admin user with required permissions
    $adminUser = $this->drupalCreateUser([
      'access content',
      'access administration pages',
      'administer content types',
      'administer node fields',
      'create article content',
      'access media overview',
      'administer media',
      'access advanced file destination',
      'create advanced file destination directories',
    ]);
    $this->drupalLogin($adminUser);

    // Set up test directories
    $test_dirs = ['public://test_uploads', 'public://test_dir'];
    foreach ($test_dirs as $dir) {
      $this->container->get('file_system')->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY);
    }

    // Add file field to article content type
    $this->createFileField('field_test_file', 'node', 'article', [
      'file_directory' => 'test_uploads',
      'uri_scheme' => 'public',
    ]);
  }

  /**
   * Tests directory selection and file upload.
   */
  public function testDirectorySelectionAndFileUpload() {
    // Create test directory
    $test_dir = 'public://test_uploads';
    $this->container->get('file_system')->prepareDirectory($test_dir, FileSystemInterface::CREATE_DIRECTORY);

    // Visit node creation page
    $this->drupalGet('node/add/article');
    $assert_session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Wait for form to be ready
    $assert_session->waitForField('Destination folder');

    // Create and attach test file
    $filename = 'test.txt';
    $file = $this->createTestFile($filename);
    $page->attachFileToField('files[field_test_file_0]', \Drupal::service('file_system')->realpath($file->getFileUri()));

    // Select directory and wait for AJAX
    $page->selectFieldOption('Destination folder', 'public://test_uploads');
    $assert_session->assertWaitOnAjaxRequest();

    // Fill required fields and submit
    $page->fillField('Title', $this->randomString());
    $page->pressButton('Save');

    // Verify file location
    $saved_file = File::load($file->id());
    $this->assertStringContainsString('test_uploads', $saved_file->getFileUri());
  }

  /**
   * Creates a test file.
   */
  protected function createTestFile($filename = NULL) {
    $filename = $filename ?: $this->randomMachineName() . '.txt';
    $filepath = 'temporary://' . $filename;
    file_put_contents($filepath, $this->randomString());

    $file = File::create([
      'uri' => $filepath,
      'filename' => $filename,
      'status' => FILE_STATUS_PERMANENT,
    ]);
    $file->save();

    return $file;
  }

  /**
   * Tests creating new directory through modal.
   */
  public function testCreateNewDirectory() {
    $this->drupalGet('node/add/article');
    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    // Wait for form to be ready
    $assert_session->waitForElement('css', 'a:contains("Create new folder")');

    // Click create new directory button and wait for modal
    $page->clickLink('Create new folder');
    $assert_session->waitForElement('css', '.ui-dialog');

    // Fill in directory name and submit
    $page->fillField('New directory name', 'new_test_dir');
    $page->pressButton('Create directory');
    $assert_session->assertWaitOnAjaxRequest();

    // Verify directory was created
    $this->assertTrue(
      file_exists('public://new_test_dir'),
      'Directory was created on filesystem'
    );

    // Verify directory appears in select list
    $assert_session->optionExists('Destination folder', 'public://new_test_dir');
  }

}
