<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\file\FileUsage\FileUsageInterface;

/**
 * Provides a form for directory deletion with file usage information.
 */
class AdvancedFileDestinationDeleteForm extends EntityDeleteForm {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new AdvancedFileDestinationDeleteForm.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    FileSystemInterface $file_system,
    FileUsageInterface $file_usage
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem = $file_system;
    $this->fileUsage = $file_usage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('file_system'),
      $container->get('file.usage')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    
    $path = $this->entity->getPath();
    
    // Build description markup
    $description = '<div class="directory-delete-confirmation">';
    $description .= '<p>' . $this->t('Are you sure you want to delete the directory %name?', [
      '%name' => $this->entity->label()
    ]) . '</p>';
    
    // Add file list first - scan recursively for all files
    $files = [];
    if (file_exists($path)) {
      $files = $this->fileSystem->scanDirectory($path, '/.*/', ['recurse' => TRUE]);
      if (!empty($files)) {
        $description .= '<div class="directory-files">';
        $description .= '<p><strong>' . $this->t('This directory contains these items:') . '</strong></p>';
        $description .= '<ul class="directory-files-list">';
        foreach ($files as $uri => $file) {
          // Check if item is directory or file
          $is_dir = is_dir($file->uri);
          $relative_path = str_replace($path . '/', '', $file->uri);
          $description .= sprintf(
            '<li class="file-item" data-type="%s"><span class="file-name">%s</span></li>',
            $is_dir ? 'directory' : 'file',
            $relative_path
          );
        }
        $description .= '</ul>';
        $description .= '</div>';
      }
    }
    
    $description .= '<p class="warning-text"><strong>' . $this->t('This action cannot be undone.') . '</strong></p>';
    $description .= '</div>';

    // Replace default description with our enhanced version
    $form['description'] = [
      '#type' => 'item',
      '#markup' => $description,
      '#allowed_tags' => ['p', 'ul', 'li', 'strong', 'div', 'span'],
    ];

    // Add custom CSS
    $form['#attached']['library'][] = 'advanced_file_destination/directory_selection';

    return $form;
  }

  /**
   * Formats a usage item for display.
   */
  protected function formatUsageItem($type, $id, $file) {
    try {
      $entity = $this->entityTypeManager->getStorage($type)->load($id);
      if ($entity) {
        $label = $entity->label() ?: $this->t('Untitled');
        $url = $entity->toUrl()->toString();
        return $this->t('@type: <a href="@url">@label</a> (using @filename)', [
          '@type' => ucfirst($type),
          '@url' => $url,
          '@label' => $label,
          '@filename' => $file->getFilename(),
        ]);
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('advanced_file_destination')->error('Error loading usage entity: @error', ['@error' => $e->getMessage()]);
    }
    return $this->t('@type #@id (using @filename)', [
      '@type' => ucfirst($type),
      '@id' => $id,
      '@filename' => $file->getFilename(),
    ]);
  }

  /**
   * Loads a file entity by URI.
   */
  protected function loadFileByUri($uri) {
    $files = $this->entityTypeManager->getStorage('file')
      ->loadByProperties(['uri' => $uri]);
    return reset($files);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $path = $this->entity->getPath();
    $success = TRUE;
    $errors = [];

    try {
      // 1. First delete files in subdirectories recursively
      $files = $this->fileSystem->scanDirectory($path, '/.*/', ['recurse' => TRUE]);
      if (!empty($files)) {
        foreach ($files as $uri => $file) {
          try {
            // Delete file entity if it exists
            $file_entity = $this->loadFileByUri($uri);
            if ($file_entity) {
              $file_entity->delete();
              \Drupal::logger('advanced_file_destination')->notice('Deleted file entity: @file', [
                '@file' => $uri
              ]);
            }
            
            // Delete physical file if it still exists
            if (file_exists($uri)) {
              $this->fileSystem->delete($uri);
              \Drupal::logger('advanced_file_destination')->notice('Deleted file: @file', [
                '@file' => $uri
              ]);
            }
          }
          catch (\Exception $e) {
            $success = FALSE;
            $errors[] = $this->t('Failed to delete file @file: @error', [
              '@file' => $uri,
              '@error' => $e->getMessage()
            ]);
          }
        }
      }

      // 2. Delete subdirectory entities recursively
      $query = $this->entityTypeManager->getStorage('afd_directory')
        ->getQuery()
        ->condition('path', $path . '/%', 'STARTS_WITH');
      $subdirectory_ids = $query->execute();

      if (!empty($subdirectory_ids)) {
        foreach ($subdirectory_ids as $id) {
          try {
            $subdirectory = $this->entityTypeManager->getStorage('afd_directory')->load($id);
            if ($subdirectory) {
              $subdirectory->delete();
              \Drupal::logger('advanced_file_destination')->notice('Deleted directory entity: @dir', [
                '@dir' => $subdirectory->getPath()
              ]);
            }
          }
          catch (\Exception $e) {
            $success = FALSE;
            $errors[] = $this->t('Failed to delete directory entity @id: @error', [
              '@id' => $id,
              '@error' => $e->getMessage()
            ]);
          }
        }
      }

      // 3. Delete the physical directory structure
      if (file_exists($path)) {
        if (!$this->fileSystem->deleteRecursive($path)) {
          throw new \Exception('Could not delete directory structure');
        }
        \Drupal::logger('advanced_file_destination')->notice('Deleted directory structure: @path', [
          '@path' => $path
        ]);
      }

      // 4. Finally delete the main directory entity
      $this->entity->delete();
      \Drupal::messenger()->addStatus($this->t('The directory %label has been deleted.', [
        '%label' => $this->entity->label(),
      ]));

      // Show any errors that occurred
      if (!empty($errors)) {
        foreach ($errors as $error) {
          \Drupal::messenger()->addWarning($error);
        }
      }

    }
    catch (\Exception $e) {
      \Drupal::messenger()->addError($this->t('Error deleting directory: @error', [
        '@error' => $e->getMessage()
      ]));
      \Drupal::logger('advanced_file_destination')->error('Directory deletion failed: @error', [
        '@error' => $e->getMessage()
      ]);
      return;
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * Scans a directory recursively for files.
   *
   * @param string $path
   *   The directory path to scan.
   *
   * @return array
   *   Array of file objects with full path information.
   */
  protected function scanDirectory($path) {
    return $this->fileSystem->scanDirectory($path, '/.*/', ['recurse' => TRUE]);
  }

}
