<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\MessageCommand;
 
/**
 * Form for creating new directories.
 */
class NewDirectoryForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_new_directory_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Directory name'),
      '#required' => TRUE,
      '#description' => $this->t('Enter the name for the new directory.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create directory'),
      '#ajax' => [
        'callback' => '::submitAjax',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Implementation handled in submitAjax.
  }

  /**
   * Ajax submit callback.
   */
  public function submitAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    
    $name = $form_state->getValue('name');
    $sanitized_name = preg_replace('/[^a-z0-9_-]/i', '_', $name);
    $date_dir = date('Y-m');
    
    $directory_service = \Drupal::service('advanced_file_destination.manager');
    $base_dir = $directory_service->getBaseDirectory();
    $new_directory = $base_dir . '/' . $date_dir . '/' . $sanitized_name;
    
    try {
      $file_system = \Drupal::service('file_system');
      
      if (!$file_system->prepareDirectory($new_directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        throw new \Exception($this->t('Failed to create directory.'));
      }
      
      // Store the new directory in tempstore
      \Drupal::service('tempstore.private')
        ->get('advanced_file_destination')
        ->set('selected_directory', $new_directory);
      
      $response->addCommand(new CloseModalDialogCommand());
      $response->addCommand(new MessageCommand($this->t('Directory created successfully.'), NULL, ['type' => 'status']));
      
      // Update directory path display
      $response->addCommand(new ReplaceCommand(
        '.directory-path-display',
        '<div class="directory-path-display visible">Created directory: ' . $new_directory . '</div>'
      ));
      
      // Update the directory selection dropdown
      $directories = \Drupal::service('advanced_file_destination.manager')->getAvailableDirectories();
      $element = [
        '#type' => 'select',
        '#options' => $directories,
        '#value' => $new_directory,
      ];
      $response->addCommand(new ReplaceCommand(
        'select[name="advanced_file_destination[directory]"]',
        \Drupal::service('renderer')->render($element)
      ));
    }
    catch (\Exception $e) {
      $response->addCommand(new MessageCommand($e->getMessage(), NULL, ['type' => 'error']));
    }
    
    return $response;
  }
}
