<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\RemoveCommand;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Form for creating a new directory in a modal.
 */
class NewDirectoryModalForm extends FormBase {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new NewDirectoryModalForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_new_directory_modal_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $parent_directory = NULL) {
    // Get field name from request if available
    $field_name = \Drupal::request()->query->get('field_name');
    if ($field_name) {
      $form_state->set('field_name', $field_name);
    }

    $form['#prefix'] = '<div id="new-directory-modal-form">';
    $form['#suffix'] = '</div>';

    // Add a hidden div that will be used as the dialog target
    $form['directory_dialog_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'afd-directory-dialog',
        'class' => ['hidden'],
      ],
    ];

    // Get parent directory from parameter first, then tempstore, then default
    if (!$parent_directory) {
      $tempstore = \Drupal::service('tempstore.private')->get('advanced_file_destination');
      $parent_directory = $tempstore->get('current_directory');
      
      if (empty($parent_directory)) {
        $parent_directory = \Drupal::service('advanced_file_destination.manager')->getDestinationDirectory();
      }
    }

    // Always ensure the parent directory is properly formatted with scheme
    $parent_directory = $this->normalizeDirectoryPath($parent_directory);

    $form['parent_directory'] = [
      '#type' => 'hidden',
      '#value' => $parent_directory,
    ];

    $form['parent_path'] = [
      '#type' => 'item',
      '#title' => $this->t('Parent directory'),
      '#markup' => '<code>' . $parent_directory . '</code>',
      '#attributes' => ['class' => ['parent-directory-display']],
    ];

    $form['directory_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('New directory name'),
      '#title_display' => 'before',
      '#required' => TRUE,
      '#maxlength' => 255,
      '#attributes' => [
        'class' => ['js-directory-name'],
        'placeholder' => $this->t('e.g., project-2024'),
      ],
    ];

    $form['preview'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['directory-preview']],
      'label' => [
        '#markup' => '<strong>' . $this->t('Full path preview:') . '</strong>',
      ],
      'path' => [
        '#prefix' => '<div class="path-preview">',
        '#markup' => $parent_directory,
        '#suffix' => '</div>',
      ],
    ];

    // Add JavaScript to update the preview
    $form['#attached']['library'][] = 'advanced_file_destination/directory_selection';

    $form['actions'] = [
      '#type' => 'actions',
    ];
    
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create directory'),
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::submitFormAjax',
        'wrapper' => 'new-directory-modal-form',
        'effect' => 'fade',
      ],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'button',
      '#value' => $this->t('Cancel'),
      '#ajax' => [
        'callback' => '::closeModal',
        'event' => 'click',
      ],
      '#attributes' => [
        'class' => ['dialog-cancel'],
      ],
    ];

    // Add our JavaScript to handle live preview
    $form['#attached']['library'][] = 'advanced_file_destination/directory_preview';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Skip validation if we're canceling
    if ($form_state->getTriggeringElement()['#id'] === 'edit-cancel') {
      $form_state->set('skip_required_validation', TRUE);
      return;
    }

    parent::validateForm($form, $form_state);
    
    // Additional validation to ensure folder name is valid
    $directory_name = $form_state->getValue('directory_name');
    if ($directory_name && preg_match('/[^a-z0-9_\-]/i', $directory_name)) {
      $form_state->setErrorByName('directory_name', $this->t('Directory name can only contain letters, numbers, underscores and hyphens.'));
    }
  }

  /**
   * Normalizes a directory path.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getDefaultDirectory();
    }
    
    // Get default directory and scheme
    $default_dir = $this->getDefaultDirectory();
    $default_scheme = 'public://';
    
    // Extract scheme from default directory
    if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
      $default_scheme = $matches[1] . '://';
    }
    
    // Ensure the path has a valid stream wrapper format
    if (strpos($path, $default_scheme) === 0) {
      // First normalize by removing any duplicate slashes after the scheme
      $path = preg_replace('#' . preg_quote($default_scheme) . '+#', $default_scheme, $path);
      
      // Fix any triple slashes that might have been created
      $path = str_replace($default_scheme . '/', $default_scheme, $path);
    }
    
    // Ensure trailing slash for directory paths
    if (substr($path, -1) !== '/') {
      $path .= '/';
    }
    
    // Log the normalized path for debugging
    \Drupal::logger('advanced_file_destination')->debug(
      'Normalized directory path: @path',
      ['@path' => $path]
    );
    
    return $path;
  }

  /**
   * Ajax submit callback.
   */
  public function submitFormAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    
    try {
      $directory_name = $form_state->getValue('directory_name');
      $parent_directory = $form_state->getValue('parent_directory');
      
      if (empty($directory_name)) {
        throw new \Exception($this->t('Directory name is required.'));
      }

      // Sanitize directory name to be filesystem-safe
      $sanitized_name = preg_replace('/[^a-z0-9_\-]/i', '_', $directory_name);
      
      // Ensure parent directory has proper scheme
      $parent_directory = $this->normalizeDirectoryPath($parent_directory);
      
      // Create full path for new directory
      $new_directory = rtrim($parent_directory, '/') . '/' . $sanitized_name;

      // Normalize the new directory path
      $new_directory = $this->normalizeDirectoryPath($new_directory);
      
      // Log the directory paths for debugging
      \Drupal::logger('advanced_file_destination')->notice('Creating new directory: @new_dir from parent: @parent_dir', [
        '@new_dir' => $new_directory,
        '@parent_dir' => $parent_directory,
      ]);

      // Get directory service
      $directory_service = \Drupal::service('advanced_file_destination.manager');

      // Ensure directory exists and is writable
      $file_system = \Drupal::service('file_system');
      if (!$file_system->prepareDirectory($new_directory, 
        FileSystemInterface::CREATE_DIRECTORY | 
        FileSystemInterface::MODIFY_PERMISSIONS
      )) {
        throw new \Exception($this->t('Failed to create directory.'));
      }

      // Create the AFD Directory entity
      $entity_values = [
        'id' => $sanitized_name,
        'name' => $directory_name,
        'path' => $new_directory,
        'status' => TRUE,
        'created' => \Drupal::time()->getRequestTime(),
        'changed' => \Drupal::time()->getRequestTime(),
      ];

      $afd_directory = \Drupal::entityTypeManager()
        ->getStorage('afd_directory')
        ->create($entity_values);

      // Set new revision
      $afd_directory->setNewRevision(TRUE);
      $afd_directory->setRevisionLogMessage($this->t('Directory created via modal form'));
      $afd_directory->setRevisionCreationTime(\Drupal::time()->getRequestTime());
      $afd_directory->setRevisionUserId(\Drupal::currentUser()->id());

      // Save the entity
      $afd_directory->save();

      // Clear cached directories and get updated list
      $directory_service->clearCachedDirectories();
      $directories = $directory_service->getAvailableDirectoriesWithTemporary($new_directory);

      // Close modal first
      $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));

      // Update select element options
      $options_html = $this->buildOptionsHtml($directories, $new_directory);

      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'empty'
      ));
      
      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'append',
        [$options_html]
      ));

      // Set the selected value
      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'val',
        [$new_directory]
      ));

      // Trigger change event to update form elements
      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'trigger',
        ['change']
      ));

      // Store the directory selection in multiple places to ensure persistence
      \Drupal::state()->set('advanced_file_destination.directory', $new_directory);
      \Drupal::service('session')->set('advanced_file_destination.directory', $new_directory);
      \Drupal::service('tempstore.private')->get('advanced_file_destination')->set('current_directory', $new_directory);

      // Force the upload location in drupalSettings
      $response->addCommand(new InvokeCommand(
        'document',
        'drupalSettings.advancedFileDestination = drupalSettings.advancedFileDestination || {}; drupalSettings.advancedFileDestination.uploadLocation = "' . $new_directory . '"'
      ));

      $response->addCommand(new MessageCommand(
        $this->t('Directory %name created successfully.', ['%name' => $directory_name]),
        NULL,
        ['type' => 'status']
      ));

    } catch (\Exception $e) {
      \Drupal::logger('advanced_file_destination')->error('Error creating directory: @error', ['@error' => $e->getMessage()]);
      $response->addCommand(new MessageCommand(
        $e->getMessage(),
        NULL,
        ['type' => 'error']
      ));
    }
    
    return $response;
  }

  /**
   * Builds the HTML for select options.
   *
   * @param array $directories
   *   Array of directories.
   * @param string $selected_value
   *   The selected directory value.
   *
   * @return string
   *   HTML string of option elements.
   */
  protected function buildOptionsHtml($directories, $selected_value) {
    $options = [];
    foreach ($directories as $value => $label) {
      $selected = $value === $selected_value ? ' selected="selected"' : '';
      $options[] = sprintf('<option value="%s"%s>%s</option>', 
        htmlspecialchars($value),
        $selected,
        htmlspecialchars($label)
      );
    }
    return implode("\n", $options);
  }

  /**
   * Renders the directory select element.
   */
  protected function renderDirectorySelect($directories, $selected) {
    $element = [
      '#type' => 'container',
      '#attributes' => ['class' => ['directory-selection-group']],
      'directory' => [
        '#type' => 'select',
        '#title' => $this->t('Destination folder'),
        '#options' => $directories,
        '#default_value' => $selected,
        '#name' => 'advanced_file_destination[directory]',
        '#description' => $this->t('Select where to store your uploaded files.'),
        '#description_display' => 'after',
        '#ajax' => [
          'callback' => 'advanced_file_destination_directory_select_callback',
          'wrapper' => 'advanced-file-destination-wrapper',
          'progress' => ['type' => 'throbber'],
        ],
      ],
      'create_directory' => [
        '#type' => 'link',
        '#title' => $this->t('Create new folder'),
        '#url' => \Drupal\Core\Url::fromRoute('advanced_file_destination.directory.modal'),
        '#attributes' => [
          'class' => ['use-ajax', 'button', 'button--small'],
          'data-dialog-type' => 'dialog',
          'data-dialog-options' => json_encode(['width' => 'auto']),
        ],
      ],
    ];

    // Use renderRoot() instead of render() for proper rendering
    return \Drupal::service('renderer')->renderRoot($element);
  }

  /**
   * AJAX callback to close the modal.
   */
  public function closeModal(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));
    // Use core RemoveCommand class
    $response->addCommand(new RemoveCommand('.messages--error'));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Handled by submitFormAjax.
  }

}
