<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Controller for the Advanced File Destination module.
 */
class AdvancedFileDestinationController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AdvancedFileDestinationController.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * Displays a list of directories.
   *
   * @return array
   *   A render array.
   */
  public function directoryList() {
    $build = [];
    
    // Add button to add new directory.
    $build['add_directory'] = [
      '#type' => 'link',
      '#title' => $this->t('Add directory'),
      '#url' => Url::fromRoute('advanced_file_destination.directory.add'),
      '#attributes' => [
        'class' => ['button', 'button--primary', 'button--action'],
      ],
    ];
    
    // Get all directories using the correct entity type ID
    $directories = $this->entityTypeManager
      ->getStorage('afd_directory')
      ->loadMultiple();
    
    // Build the table.
    $build['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Name'),
        $this->t('Path'),
        $this->t('Status'),
        $this->t('Operations'),
      ],
      '#rows' => [],
      '#empty' => $this->t('No directories available. <a href="@url">Add a directory</a>.', [
        '@url' => Url::fromRoute('advanced_file_destination.directory.add')->toString(),
      ]),
    ];
    
    foreach ($directories as $directory) {
      $build['table']['#rows'][] = [
        'data' => [
          'name' => $directory->label(),
          'path' => $directory->getPath(),
          'status' => $directory->isActive() ? $this->t('Active') : $this->t('Inactive'),
          'operations' => [
            'data' => [
              '#type' => 'operations',
              '#links' => [
                'edit' => [
                  'title' => $this->t('Edit'),
                  'url' => $directory->toUrl('edit-form'),
                ],
                'delete' => [
                  'title' => $this->t('Delete'),
                  'url' => $directory->toUrl('delete-form'),
                ],
              ],
            ],
          ],
        ],
      ];
    }
    
    // Add filesystem information section
    $build['filesystem'] = [
      '#type' => 'details',
      '#title' => $this->t('Filesystem information'),
      '#open' => FALSE,
      'info' => [
        '#markup' => '<p>' . $this->t('The following directories are available in the filesystem:') . '</p>',
      ],
    ];
    
    // Get the configured stream wrappers
    $config = $this->configFactory->get('advanced_file_destination.settings');
    $default_directory = $config->get('default_directory') ?: 'public://';
    $stream_wrappers = [];
    
    // Always include the default directory
    $scheme = '';
    if (preg_match('#^([a-z]+://)#', $default_directory, $matches)) {
      $scheme = $matches[1];
      $stream_wrappers[] = $scheme;
    }
    
    // Check if other stream wrappers are enabled
    if ($config->get('include_public')) {
      $stream_wrappers[] = 'public://';
    }
    if ($config->get('include_private')) {
      $stream_wrappers[] = 'private://';
    }
    if ($config->get('include_assets')) {
      $stream_wrappers[] = 'assets://';
    }
    if ($config->get('include_temporary')) {
      $stream_wrappers[] = 'temporary://';
    }
    
    // Deduplicate stream wrappers
    $stream_wrappers = array_unique($stream_wrappers);
    
    // If no stream wrappers specified, fall back to public and private
    if (empty($stream_wrappers)) {
      $stream_wrappers = ['public://', 'private://'];
    }
    
    // Scan the filesystem for directories.
    $file_system = \Drupal::service('file_system');
    $directories = [];
    
    foreach ($stream_wrappers as $root) {
      if (file_exists($root)) {
        $scanned = $file_system->scanDirectory($root, '/^.+$/', ['recurse' => FALSE, 'key' => 'uri']);
        foreach ($scanned as $uri => $file) {
          if (is_dir($uri)) {
            $directories[] = $uri;
          }
        }
      }
    }
    
    $build['filesystem']['list'] = [
      '#theme' => 'item_list',
      '#items' => $directories,
      '#empty' => $this->t('No directories found in the filesystem.'),
    ];
    
    return $build;
  }
 
}
