<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a listing of directories with enhanced UI features.
 */
class DirectoryListBuilder extends EntityListBuilder implements FormInterface {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * {@inheritdoc}
   */
  protected $limit = 25;

  /**
   * Constructs a new DirectoryListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, FormBuilderInterface $form_builder) {
    parent::__construct($entity_type, $storage);
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('form_builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'directory_list_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Add the library
    $form['#attached']['library'][] = 'advanced_file_destination/directory_list';
    
    // Add filters
    $form['filters'] = [
      '#type' => 'details',
      '#title' => $this->t('Filters'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['directory-filters', 'clearfix']],
      'status' => [
        '#type' => 'select',
        '#title' => $this->t('Status'),
        '#options' => [
          '' => $this->t('- Any -'),
          '1' => $this->t('Active'),
          '0' => $this->t('Disabled'),
        ],
        '#default_value' => \Drupal::request()->query->get('status'),
      ],
      
      'name' => [
        '#type' => 'search',
        '#title' => $this->t('Search by name'),
        '#default_value' => \Drupal::request()->query->get('name'),
        '#size' => 30,
      ],
      'actions' => [
        '#type' => 'actions',
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Filter'),
          '#button_type' => 'primary',
        ],
        'reset' => [
          '#type' => 'link',
          '#title' => $this->t('Reset'),
          '#url' => Url::fromRoute('<current>'),
          '#attributes' => ['class' => ['button']],
        ],
      ],
    ];

    // Build table
    $form['directories'] = [
      '#type' => 'details',
      '#title' => $this->t('Directories'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['directory-listing']],
    ];
    $form['directories']['table'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#empty' => $this->t('No directories available.'),
      '#tableselect' => TRUE,
      '#attributes' => ['class' => ['directory-listing-table']],
    ];

    // Load entities with pagination
    $entities = $this->load();
    foreach ($entities as $entity) {
      $form['directories']['table'][$entity->id()] = $this->buildRow($entity);
    }

    // Add bulk operations with new structure
    $form['directories']['operations'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['directory-bulk-operations'],
        'data-drupal-selector' => 'edit-bulk-actions',
        'role' => 'group',
      ],
      'status' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['directory-bulk-operations__item', 'directory-bulk-operations__item--status'],
        ],
        '#markup' => $this->t('No items selected'),
      ],
      'action' => [
        '#type' => 'select',
        '#title' => $this->t('Action'),
        '#title_display' => 'invisible',
        '#options' => [
          'enable' => $this->t('Enable'),
          'disable' => $this->t('Disable'),
          'delete' => $this->t('Delete'),
        ],
        '#empty_option' => $this->t('- Select action -'),
        '#wrapper_attributes' => [
          'class' => ['directory-bulk-operations__item', 'directory-bulk-operations__item--preceding-actions'],
        ],
      ],
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Apply to selected items'),
        '#button_type' => 'primary',
        '#attributes' => ['class' => ['button--small']],
        '#wrapper_attributes' => ['class' => ['directory-bulk-operations__item']],
      ],
    ];

    // Add pager
    $form['pager'] = [
      '#type' => 'pager',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $filters = [];
    foreach (['status', 'name'] as $filter) {
      if ($value = $form_state->getValue($filter)) {
        $filters[$filter] = $value;
      }
    }
    $form_state->setRedirect('<current>', $filters);
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header = [
      'preview' => [
        'data' => $this->t('Preview'),
        'class' => ['directory-preview-header'],
      ]
    ];
    
    $header += [
      'name' => [
        'data' => $this->t('Name'),
        'field' => 'label',
        'specifier' => 'label',
        'sort' => 'asc',
      ],
      'path' => [
        'data' => $this->t('Path'),
        'field' => 'path',
        'specifier' => 'path',
      ],
      'status' => [
        'data' => $this->t('Status'),
        'field' => 'status',
        'specifier' => 'status',
      ],
      'created' => [
        'data' => $this->t('Created'),
        'field' => 'created',
        'specifier' => 'created',
        'sort' => 'desc',
      ],
      'changed' => [
        'data' => $this->t('Updated'),
        'field' => 'changed',
        'specifier' => 'changed',
        'sort' => 'desc',
      ],
      'operations' => [
        'data' => $this->t('Operations'),
      ],
    ];
    
    return $header;
  }

  /**
   * {@inheritdoc}
   */
  public function load() {
    $query = $this->getStorage()->getQuery()
      ->accessCheck(TRUE)
      ->pager($this->limit);

    // Add filters
    if ($status = \Drupal::request()->query->get('status')) {
      $query->condition('status', $status);
    }
    if ($name = \Drupal::request()->query->get('name')) {
      $query->condition('name', '%' . $name . '%', 'LIKE');
    }

    // Get sort parameters from URL or use defaults
    $header = $this->buildHeader();
    $order = \Drupal::request()->query->get('order');
    $sort = \Drupal::request()->query->get('sort');

    // If no sort specified in URL, use changed/DESC as default
    if (empty($sort)) {
      $sort = 'changed';
      $order = 'DESC';
    }

    // Ensure we have a valid sort field
    if (!isset($header[$sort]['field'])) {
      $sort = 'changed';
      $order = 'DESC';
    }

    // Apply sorting
    $query->sort($header[$sort]['field'], $order);

    $ids = $query->execute();
    return $this->storage->loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    return $this->formBuilder->getForm($this);
  }

  /**
   * Validate handler for bulk operations.
   */
  public function validateBulkForm(array &$form, FormStateInterface $form_state) {
    $selected = array_filter($form_state->getValue('table'));
    if (empty($selected)) {
      $form_state->setError($form['directories']['operations'], $this->t('No items selected.'));
    }
  }

  /**
   * Submit handler for bulk operations.
   */
  public function submitBulkForm(array &$form, FormStateInterface $form_state) {
    $selected = array_filter($form_state->getValue('table'));
    $action = $form_state->getValue('action');
    $entities = $this->storage->loadMultiple($selected);

    switch ($action) {
      case 'enable':
      case 'disable':
        foreach ($entities as $entity) {
          $entity->set('status', $action === 'enable')->save();
        }
        $this->messenger()->addStatus($this->t('@count items have been @action.', [
          '@count' => count($selected),
          '@action' => $action . 'd',
        ]));
        break;

      case 'delete':
        try {
          $this->storage->delete($entities);
          $this->messenger()->addStatus($this->t('@count items have been deleted.', [
            '@count' => count($selected),
          ]));
        }
        catch (\Exception $e) {
          $this->messenger()->addError($this->t('Error deleting items: @error', [
            '@error' => $e->getMessage(),
          ]));
        }
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getTriggeringElement()['#parents'][0] === 'operations') {
      // Validate bulk operations
      $selected = array_filter($form_state->getValue('table', []));
      if (empty($selected)) {
        $form_state->setErrorByName('operations', $this->t('No items selected.'));
      }
      
      $action = $form_state->getValue(['operations', 'action']);
      if (empty($action)) {
        $form_state->setErrorByName('operations[action]', $this->t('Please select an operation.'));
      }
    }
    else {
      // Validate filters
      if ($name = $form_state->getValue('name')) {
        if (strlen($name) < 2) {
          $form_state->setErrorByName('name', $this->t('Search term must be at least 2 characters.'));
        }
      }
    }
  }

  /**
   * Builds a row for the directory list.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return array
   *   A render array representing the row.
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $entity */
    $row = [];

    // Add preview column with folder icon
    $row['preview'] = [
      'data' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['directory-preview-wrapper']],
        'icon' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#attributes' => [
            'class' => ['directory-icon'],
            'title' => $this->t('Directory'),
          ],
        ],
      ],
    ];

    // Check if path exists and is a directory
    $path = $entity->getPath();
    $is_directory = file_exists($path) && is_dir($path);

    $row['name'] = [
      'data' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['directory-name-wrapper']],
        'icon' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#attributes' => [
            'class' => ['directory-icon'],
            'title' => $this->t('Directory'),
          ],
        ],
        'link' => [
          '#type' => 'link',
          '#title' => $entity->getName(),
          '#url' => $is_directory ? 
            Url::fromRoute('entity.afd_directory.edit_form', ['afd_directory' => $entity->id()]) : 
            $entity->toUrl('edit-form'),
        ],
      ],
    ];

    $row['path'] = [
      'data' => ['#markup' => $entity->getPath()],
    ];

    $row['status'] = [
      'data' => [
        '#markup' => $entity->isActive() ? $this->t('Active') : $this->t('Disabled'),
        '#wrapper_attributes' => [
          'class' => ['directory-status', $entity->isActive() ? 'directory-status--active' : 'directory-status--disabled'],
        ],
      ],
    ];

    $row['created'] = [
      'data' => [
        '#markup' => \Drupal::service('date.formatter')->format($entity->getCreatedTime()),
      ],
    ];

    $row['changed'] = [
      'data' => [
        '#markup' => $entity->getChangedTime() ? \Drupal::service('date.formatter')->format($entity->getChangedTime()) : \Drupal::service('date.formatter')->format($entity->getCreatedTime()),
      ],
    ];

    $row['operations'] = [
      'data' => $this->buildOperations($entity),
    ];

    return $row;
  }

}
