<?php

namespace Drupal\advanced_file_destination\EventSubscriber;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\FileInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Event subscriber for advanced file destination.
 */
class AdvancedFileDestinationSubscriber implements EventSubscriberInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AdvancedFileDestinationSubscriber.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The advanced file destination manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    AccountProxyInterface $current_user, 
    AdvancedFileDestinationManager $destination_manager,
    ConfigFactoryInterface $config_factory
  ) {
    $this->currentUser = $current_user;
    $this->destinationManager = $destination_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'file.presave' => ['onFilePresave', 200], // Higher priority to run first
    ];
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Reacts to the file presave event.
   *
   * @param $event
   *   The file presave event.
   */
  public function onFilePresave($event) {
    $file = $event->getFile();
    
    if (!$file->isNew()) {
      return;
    }

    try {
      // Get directory from state - this is now authoritative
      $selected_directory = \Drupal::state()->get('advanced_file_destination.directory');
      
      if (!$selected_directory) {
        $directory_service = \Drupal::service('advanced_file_destination.manager');
        $selected_directory = $directory_service->getCurrentDirectory();
      }

      // Get default directory from settings
      $default_directory = $this->getDefaultDirectory();

      if (!$selected_directory || $selected_directory === $default_directory) {
        return;
      }

      $current_uri = $file->getFileUri();
      if (!$current_uri) {
        return;
      }

      // Don't move if already in correct directory
      if (strpos($current_uri, $selected_directory) === 0) {
        return;
      }

      $file_system = \Drupal::service('file_system');
      $filename = basename($current_uri);
      $new_uri = rtrim($selected_directory, '/') . '/' . $filename;

      // Handle duplicates using FileSystemInterface
      if (file_exists($new_uri)) {
        $new_uri = $file_system->getDestinationFilename($new_uri, FileSystemInterface::EXISTS_RENAME);
      }

      // Move the file using FileSystem service
      $directory = dirname($new_uri);
      if (!$file_system->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        throw new \Exception('Directory not writable: ' . $directory);
      }

      // Move the file only if it exists
      if (file_exists($current_uri)) {
        if (!$file_system->move($current_uri, $new_uri, FileSystemInterface::EXISTS_RENAME)) {
          throw new \Exception('Failed to move file to ' . $new_uri);
        }
      }

      // Update the file entity
      $file->setFileUri($new_uri);
      $file->setPermanent();

      \Drupal::logger('advanced_file_destination')->notice(
        'File moved during presave: @new (from: @old)', 
        ['@new' => $new_uri, '@old' => $current_uri]
      );

    } catch (\Exception $e) {
      \Drupal::logger('advanced_file_destination')->error(
        'File move failed: @error', 
        ['@error' => $e->getMessage()]
      );
    }
  }

}
