<?php

namespace Drupal\advanced_file_destination\EventSubscriber;

use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * File pre-render event subscriber.
 */
class FilePreRenderSubscriber implements TrustedCallbackInterface {

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Static instance for static callbacks.
   *
   * @var \Drupal\advanced_file_destination\EventSubscriber\FilePreRenderSubscriber
   */
  protected static $instance;

  /**
   * Constructs a new FilePreRenderSubscriber.
   *
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The advanced file destination manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    AdvancedFileDestinationManager $destination_manager,
    ConfigFactoryInterface $config_factory
  ) {
    $this->destinationManager = $destination_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * Gets the static instance.
   *
   * @return \Drupal\advanced_file_destination\EventSubscriber\FilePreRenderSubscriber
   *   The static instance.
   */
  public static function getInstance() {
    if (!static::$instance) {
      static::$instance = \Drupal::service('advanced_file_destination.file_pre_render');
    }
    return static::$instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRenderCallback'];
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Gets the default directory statically.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected static function getStaticDefaultDirectory() {
    return \Drupal::config('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Static pre-render callback.
   */
  public static function preRenderCallback(array $element) {
    // Get directory from state first - this is now authoritative
    $selected_directory = \Drupal::state()->get('advanced_file_destination.directory');
    
    if (!$selected_directory) {
      $directory_service = \Drupal::service('advanced_file_destination.manager');
      $selected_directory = $directory_service->getCurrentDirectory();
    }

    // Fix public directory paths to ensure they are properly formatted
    if ($selected_directory === 'public:/' || $selected_directory === 'public://public:/' || 
        $selected_directory === 'public://public://' || $selected_directory === 'public:///') {
      $selected_directory = 'public://';
    }

    // Get default directory from configuration
    $default_directory = self::getStaticDefaultDirectory();

    if ($selected_directory) {
      // Force directory in all possible locations
      $element['#upload_location'] = $selected_directory;
      $element['#file_directory'] = $selected_directory;
      $element['#upload_validators']['file_validator'] = [
        'function' => 'advanced_file_destination_validate_file_directory',
        'directory' => $selected_directory,
      ];

      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $selected_directory;
        $element['upload']['#file_directory'] = $selected_directory;
        $element['upload']['#upload_validators'] = $element['#upload_validators'];
      }

      // Force static defaults to maintain directory during upload
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $selected_directory;
      $settings['file_directory'] = $selected_directory;
      
      // Store in state for immediate use
      \Drupal::state()->set('advanced_file_destination.directory', $selected_directory);
      \Drupal::state()->set('advanced_file_destination.directory_timestamp', time());

      // Add JavaScript settings
      $element['#attached']['drupalSettings']['advancedFileDestination'] = [
        'uploadLocation' => $selected_directory,
      ];

      \Drupal::logger('advanced_file_destination')->debug(
        'Pre-render enforcing upload location: @dir', 
        ['@dir' => $selected_directory]
      );
    }

    return $element;
  }

  /**
   * Pre-render callback for file elements.
   *
   * @param array $element
   *   The render element to process.
   *
   * @return array
   *   The processed element.
   */
  public function preRender(array $element) {
    if ($selected_directory = $this->destinationManager->getDestinationDirectory()) {
      $element['#upload_location'] = $selected_directory;
      
      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $selected_directory;
      }

      \Drupal::logger('advanced_file_destination')->notice(
        'Pre-render updated upload location to: @dir',
        ['@dir' => $selected_directory]
      );
    }
    
    return $element;
  }

}
