<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\RemoveCommand;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Drupal\Core\State\StateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Symfony\Component\HttpFoundation\RequestStack;
use Exception;

/**
 * Form for creating a new directory in a modal.
 */
class NewDirectoryModalForm extends FormBase {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The private tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $afdManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $timeService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The session manager.
   *
   * @var \Symfony\Component\HttpFoundation\Session\Session
   */
  protected $sessionManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new NewDirectoryModalForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $afd_manager
   *   The advanced file destination manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system.
   * @param \Drupal\Component\Datetime\TimeInterface $time_service
   *   The time service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Symfony\Component\HttpFoundation\Session\Session $session_manager
   *   The session manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RequestStack $request_stack,
    PrivateTempStoreFactory $temp_store_factory,
    AdvancedFileDestinationManager $afd_manager,
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system,
    TimeInterface $time_service,
    EntityTypeManagerInterface $entity_type_manager,
    StateInterface $state,
    Session $session_manager,
    RendererInterface $renderer,
    AccountProxyInterface $current_user
  ) {
    $this->configFactory = $config_factory;
    $this->requestStack = $request_stack;
    $this->tempStoreFactory = $temp_store_factory;
    $this->afdManager = $afd_manager;
    $this->loggerFactory = $logger_factory;
    $this->fileSystem = $file_system;
    $this->timeService = $time_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
    $this->sessionManager = $session_manager;
    $this->renderer = $renderer;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('request_stack'),
      $container->get('tempstore.private'),
      $container->get('advanced_file_destination.manager'),
      $container->get('logger.factory'),
      $container->get('file_system'),
      $container->get('datetime.time'),
      $container->get('entity_type.manager'),
      $container->get('state'),
      $container->get('session'),
      $container->get('renderer'),
      $container->get('current_user')
    );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_new_directory_modal_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $parent_directory = NULL) {
    // Get field name from request if available.
    $field_name = $this->requestStack->getCurrentRequest()->query->get('field_name');
    if ($field_name) {
      $form_state->set('field_name', $field_name);
    }

    $form['#prefix'] = '<div id="new-directory-modal-form">';
    $form['#suffix'] = '</div>';

    // Add a hidden div that will be used as the dialog target.
    $form['directory_dialog_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'afd-directory-dialog',
        'class' => ['hidden'],
      ],
    ];
      
    $parent_directory = $this->afdManager->getDestinationDirectory();
    

    // Normalize the parent directory path.
    $parent_directory = $this->normalizeDirectoryPath($parent_directory);

    $form['parent_directory'] = [
      '#type' => 'hidden',
      '#value' => $parent_directory,
    ];

    $form['parent_path'] = [
      '#type' => 'item',
      '#title' => $this->t('Parent directory'),
      '#markup' => '<code>' . $parent_directory . '</code>',
      '#attributes' => ['class' => ['parent-directory-display']],
    ];

    $form['directory_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('New directory name'),
      '#title_display' => 'before',
      '#required' => TRUE,
      '#maxlength' => 255,
      '#attributes' => [
        'class' => ['js-directory-name'],
        'placeholder' => $this->t('e.g., project-2024'),
      ],
    ];

    $form['preview'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['directory-preview']],
      'label' => [
        '#markup' => '<strong>' . $this->t('Full path preview:') . '</strong>',
      ],
      'path' => [
        '#prefix' => '<div class="path-preview">',
        '#markup' => $parent_directory,
        '#suffix' => '</div>',
      ],
    ];

    // Add JavaScript to update the preview.
    $form['#attached']['library'][] = 'advanced_file_destination/directory_selection';

    $form['actions'] = [
      '#type' => 'actions',
    ];
    
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create directory'),
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::submitFormAjax',
        'wrapper' => 'new-directory-modal-form',
        'effect' => 'fade',
      ],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'button',
      '#value' => $this->t('Cancel'),
      '#ajax' => [
        'callback' => '::closeModal',
        'event' => 'click',
      ],
      '#attributes' => [
        'class' => ['dialog-cancel'],
      ],
    ];

    // Add our JavaScript to handle live preview.
    $form['#attached']['library'][] = 'advanced_file_destination/directory_preview';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Skip validation if we're canceling.
    if ($form_state->getTriggeringElement()['#id'] === 'edit-cancel') {
      $form_state->set('skip_required_validation', TRUE);
      return;
    }

    parent::validateForm($form, $form_state);
    
    // Additional validation to ensure folder name is valid.
    $directory_name = $form_state->getValue('directory_name');
    if ($directory_name && preg_match('/[^a-z0-9_\-]/i', $directory_name)) {
      $form_state->setErrorByName('directory_name', $this->t('Directory name can only contain letters, numbers, underscores and hyphens.'));
    }
  }

  /**
   * Gets valid stream wrappers from configuration.
   *
   * @return array
   *   Array of valid stream wrapper schemes.
   */
  protected function getValidStreamWrappers() {
    $config = $this->configFactory->get('advanced_file_destination.settings');
    $wrappers = [];
    
    // Add all configured stream wrappers.
    if ($config->get('include_public')) {
      $wrappers[] = 'public';
    }
    if ($config->get('include_private')) {
      $wrappers[] = 'private';
    }
    if ($config->get('include_assets')) {
      $wrappers[] = 'assets';
    }
    if ($config->get('include_temporary')) {
      $wrappers[] = 'temporary';
    }
    
    // If no wrappers are configured, default to public and private.
    if (empty($wrappers)) {
      $wrappers = ['public', 'private'];
    }
    
    return $wrappers;
  }

  /**
   * Normalizes a directory path.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getDefaultDirectory();
    }

    if ($path === 'public:/') {
      return 'public://';
    }
    // Get valid stream wrappers.
    $valid_wrappers = $this->getValidStreamWrappers();
    $valid_wrappers_regex = implode('|', $valid_wrappers);
    
    // If already has a valid scheme, return as is.
    if (preg_match('#^(' . $valid_wrappers_regex . ')://#', $path)) {
      // Remove any duplicate scheme.
      $path = preg_replace('#^(' . $valid_wrappers_regex . ')://((' . $valid_wrappers_regex . ')://)#', '$1://', $path);
      return $path;
    }

    // Get the default directory scheme.
    $default_dir = $this->getDefaultDirectory();
    $default_scheme = 'public://';
    
    // Extract scheme from default directory.
    if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
      $default_scheme = $matches[1] . '://';
    }

    // Add scheme if missing.
    return $default_scheme . ltrim($path, '/');
  }

  /**
   * Ajax submit callback.
   */
  public function submitFormAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    
    try {
      $directory_name = $form_state->getValue('directory_name');
      $parent_directory = $form_state->getValue('parent_directory');
      
      if (empty($directory_name)) {
        // Use untranslated exception message, will be translated for display.
        throw new Exception('Directory name is required.');
      }

      // Sanitize directory name to be filesystem-safe.
      $sanitized_name = preg_replace('/[^a-z0-9_\-]/i', '_', $directory_name);
      
      // Ensure parent directory has proper scheme.
      $parent_directory = $this->normalizeDirectoryPath($parent_directory);
      
      // Create full path for new directory.
      $new_directory = rtrim($parent_directory, '/') . '/' . $sanitized_name;

      // Normalize the new directory path.
      $new_directory = $this->normalizeDirectoryPath($new_directory);
      
      // Log the directory paths for debugging.
      $this->loggerFactory->get('advanced_file_destination')->notice('Creating new directory: @new_dir from parent: @parent_dir', [
        '@new_dir' => $new_directory,
        '@parent_dir' => $parent_directory,
      ]);

      // Ensure directory exists and is writable.
      if (!$this->fileSystem->prepareDirectory($new_directory, 
        FileSystemInterface::CREATE_DIRECTORY | 
        FileSystemInterface::MODIFY_PERMISSIONS
      )) {
        // Use untranslated exception message, will be translated for display.
        throw new Exception('Failed to create directory.');
      }

      // Create the AFD Directory entity.
      $entity_values = [
        'id' => $sanitized_name,
        'name' => $directory_name,
        'path' => $new_directory,
        'status' => TRUE,
        'created' => $this->timeService->getRequestTime(),
        'changed' => $this->timeService->getRequestTime(),
      ];

      $afd_directory = $this->entityTypeManager
        ->getStorage('afd_directory')
        ->create($entity_values);

      // Set new revision.
      $afd_directory->setNewRevision(TRUE);
      $afd_directory->setRevisionLogMessage($this->t('Directory created via modal form'));
      $afd_directory->setRevisionCreationTime($this->timeService->getRequestTime());
      $afd_directory->setRevisionUserId($this->currentUser->id());

      // Save the entity.
      $afd_directory->save();

      // Clear cached directories and get updated list.
      $this->afdManager->clearCachedDirectories();
      $directories = $this->afdManager->getAvailableDirectoriesWithTemporary($new_directory);

      // Close modal first.
      $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));

      // Update select element options.
      $options_html = $this->buildOptionsHtml($directories, $new_directory);

      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'empty'
      ));
      
      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'append',
        [$options_html]
      ));

      // Set the selected value.
      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'val',
        [$new_directory]
      ));

      // Trigger change event to update form elements.
      $response->addCommand(new InvokeCommand(
        'select.js-advanced-file-destination-select',
        'trigger',
        ['change']
      ));

      // Store the directory selection in multiple places to ensure persistence.
      $this->state->set('advanced_file_destination.directory', $new_directory);
      $this->sessionManager->set('advanced_file_destination.directory', $new_directory);
      $this->tempStoreFactory->get('advanced_file_destination')->set('current_directory', $new_directory);

      // Force the upload location in drupalSettings.
      $response->addCommand(new InvokeCommand(
        'document',
        'drupalSettings.advancedFileDestination = drupalSettings.advancedFileDestination || {}; drupalSettings.advancedFileDestination.uploadLocation = "' . $new_directory . '"'
      ));

      $response->addCommand(new MessageCommand(
        $this->t('Directory %name created successfully.', ['%name' => $directory_name]),
        NULL,
        ['type' => 'status']
      ));

    }
    catch (Exception $e) {
      $this->loggerFactory->get('advanced_file_destination')->error('Error creating directory: @error', ['@error' => $e->getMessage()]);
      // Translate exception message for display.
      $response->addCommand(new MessageCommand(
        $this->t($e->getMessage()),
        NULL,
        ['type' => 'error']
      ));
    }
    
    return $response;
  }

  /**
   * Builds the HTML for select options.
   *
   * @param array $directories
   *   Array of directories.
   * @param string $selected_value
   *   The selected directory value.
   *
   * @return string
   *   HTML string of option elements.
   */
  protected function buildOptionsHtml($directories, $selected_value) {
    $options = [];
    foreach ($directories as $value => $label) {
      $selected = $value === $selected_value ? ' selected="selected"' : '';
      $options[] = sprintf('<option value="%s"%s>%s</option>', 
        htmlspecialchars($value),
        $selected,
        htmlspecialchars($label)
      );
    }
    return implode("\n", $options);
  }

  /**
   * Renders the directory select element.
   */
  protected function renderDirectorySelect($directories, $selected) {
    $element = [
      '#type' => 'container',
      '#attributes' => ['class' => ['directory-selection-group']],
      'directory' => [
        '#type' => 'select',
        '#title' => $this->t('Destination folder'),
        '#options' => $directories,
        '#default_value' => $selected,
        '#name' => 'advanced_file_destination[directory]',
        '#description' => $this->t('Select where to store your uploaded files.'),
        '#description_display' => 'after',
        '#ajax' => [
          'callback' => 'advanced_file_destination_directory_select_callback',
          'wrapper' => 'advanced-file-destination-wrapper',
          'progress' => ['type' => 'throbber'],
        ],
      ],
      'create_directory' => [
        '#type' => 'link',
        '#title' => $this->t('Create new folder'),
        '#url' => Url::fromRoute('advanced_file_destination.directory.modal'),
        '#attributes' => [
          'class' => ['use-ajax', 'button', 'button--small'],
          'data-dialog-type' => 'dialog',
          'data-dialog-options' => json_encode(['width' => 'auto']),
        ],
      ],
    ];

    // Use renderRoot() instead of render() for proper rendering.
    return $this->renderer->renderRoot($element);
  }

  /**
   * AJAX callback to close the modal.
   */
  public function closeModal(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));
    // Use core RemoveCommand class.
    $response->addCommand(new RemoveCommand('.messages--error'));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Handled by submitFormAjax.
  }

}
