<?php

namespace Drupal\advanced_file_destination\Service;

use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Handles file element callbacks.
 */
class FileElementHandler implements TrustedCallbackInterface {

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * Constructs a new FileElementHandler.
   *
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The advanced file destination manager.
   */
  public function __construct(AdvancedFileDestinationManager $destination_manager) {
    $this->destinationManager = $destination_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRenderFileElement'];
  }

  /**
   * Pre-render callback for file elements.
   *
   * @param array $element
   *   The element to pre-render.
   *
   * @return array
   *   The processed element.
   */
  public function preRenderFileElement(array $element) {
    if ($selected_directory = $this->destinationManager->getDestinationDirectory()) {
      // Set upload location for all possible elements
      $element['#upload_location'] = $selected_directory;
      
      // Ensure directory selection persists
      if (isset($element['directory']) && !isset($element['directory']['#default_value'])) {
        $element['directory']['#default_value'] = $selected_directory;
      }

      // Update form values
      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $selected_directory;
        $element['upload']['#upload_validators']['file_directory'] = $selected_directory;
      }

      // Set static file upload defaults
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $selected_directory;

      // Add settings to element for JavaScript
      $element['#attached']['drupalSettings']['advancedFileDestination']['currentDirectory'] = $selected_directory;
    }

    return $element;
  }

}
