<?php

namespace Drupal\Tests\advanced_file_destination\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\file\Entity\File;
use Drupal\Core\File\FileSystemInterface;

/**
 * Tests the Advanced File Destination functionality.
 *
 * @group advanced_file_destination
 */
class AdvancedFileDestinationTest extends KernelTestBase {

  protected static $modules = [
    'system',
    'file',
    'user',
    'advanced_file_destination',
  ];

  protected $destinationManager;
  protected $fileSystem;

  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    
    $this->destinationManager = $this->container->get('advanced_file_destination.manager');
    $this->fileSystem = $this->container->get('file_system');
    
    // Create test directories
    $test_directories = [
      'public://test_directory',
      'public://images',
      'public://documents',
    ];
    
    foreach ($test_directories as $directory) {
      $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);
    }
  }

  public function testFileUploadLocation() {
    // Set the destination directory
    $test_directory = 'public://test_directory';
    $this->destinationManager->setDestinationDirectory($test_directory);

    // Create a test file
    $filename = $this->randomMachineName() . '.txt';
    $contents = $this->randomString();
    $original_uri = 'temporary://' . $filename;
    file_put_contents($original_uri, $contents);

    // Create file entity
    $file = File::create([
      'uri' => $original_uri,
      'filename' => $filename,
      'status' => 1,
    ]);
    $file->save();

    // Trigger pre-save operations
    advanced_file_destination_file_pre_save($file);

    // Verify file location
    $expected_uri = $test_directory . '/' . $filename;
    $this->assertEquals($expected_uri, $file->getFileUri(), 'File was not moved to the correct directory.');
    $this->assertTrue(file_exists($expected_uri), 'File does not exist in the target location.');
  }

  public function testDirectorySelection() {
    // Test initial directory selection
    $test_directory = 'public://images';
    $this->destinationManager->setDestinationDirectory($test_directory);
    $this->assertEquals($test_directory, $this->destinationManager->getDestinationDirectory());

    // Test directory change
    $new_directory = 'public://documents';
    $this->destinationManager->setDestinationDirectory($new_directory);
    $this->assertEquals($new_directory, $this->destinationManager->getDestinationDirectory());
  }
}
