(function ($, Drupal, once) {
  'use strict';

  let isProcessing = false;
  let currentDirectory = '';
  let lastValidDirectory = '';

  function updateFormElements(directory) {
    if (!directory || isProcessing) { return;
    }

    isProcessing = true;

    // Update all directory selects with consistent class and structure
    const $selects = $('select[data-drupal-selector="edit-field-media-image-directory"], .js-advanced-file-destination-select');
    $selects.each(function () {
      const $select = $(this);
      if ($select.val() !== directory) {
        $select.val(directory).trigger('change');
      }
    });

    // Ensure directory selection persists
    currentDirectory = directory;
    lastValidDirectory = directory;

    // Update managed file elements
    $('.js-form-managed-file').each(function () {
      $(this).attr('data-upload-location', directory);
      $(this).find('input[type="file"]').attr('data-upload-location', directory);
    });

    // Store in drupalSettings
    if (drupalSettings.advancedFileDestination) {
      drupalSettings.advancedFileDestination.uploadLocation = directory;
    }

    // Send AJAX request to update server state
    const formData = new FormData();
    formData.append('directory', directory);

    fetch(Drupal.url('advanced-file-destination/ajax/update-state'), {
      method: 'POST',
      body: formData,
      credentials: 'same-origin',
      headers: {
        'X-CSRF-Token': drupalSettings.advanced_file_destination ? drupalSettings.advanced_file_destination.csrfToken : '',
      }
    })
    .then(response => response.json())
    .then(data => {
      if (data.status === 'success') {
        console.log('Directory updated:', data.directory);
      } else {
        console.error('Error updating directory:', data.message);
      }
      isProcessing = false;
    })
    .catch(error => {
      console.error('Error updating directory state:', error);
      isProcessing = false;
    });

    // Trigger update for any media library fields
    triggerFormUpdates(directory);

    setTimeout(() => {
      isProcessing = false;
    }, 300);
  }

  function triggerFormUpdates(directory) {
    // Trigger native change events
    $('.js-advanced-file-destination-select').each(function () {
      this.dispatchEvent(new Event('change', { bubbles: true }));
    });

    // Update any media library fields
    $('.media-library-add-form input[type="file"]').each(function () {
      const $form = $(this).closest('form');
      const $uploadLocation = $form.find('[data-drupal-selector="edit-upload-location"]');
      if ($uploadLocation.length) {
        $uploadLocation.val(directory).trigger('change');
      }
    });

    // Force update for hidden media upload location fields
    $('input[name="upload_location"]').val(directory);
  }

  Drupal.behaviors.directorySelection = {
    attach: function (context, settings) {
      // Initialize from settings
      if (settings.advancedFileDestination && settings.advancedFileDestination.uploadLocation && !currentDirectory) {
        currentDirectory = settings.advancedFileDestination.uploadLocation;
        lastValidDirectory = currentDirectory;
      }

      // Ensure all directory selects have the required class
      once('directory-selection-class', 'select[name="advanced_file_destination[directory]"]', context).forEach(function (element) {
        $(element).addClass('js-advanced-file-destination-select');
      });

      // Use consistent selector for directory select elements
      once('directory-selection', 'select[name="advanced_file_destination[directory]"]', context).forEach(function (element) {
        const $element = $(element);
        $element.addClass('js-advanced-file-destination-select');
        $element.on('change', function (e) {
          const directory = $(this).val();
          if (directory && directory !== currentDirectory && !isProcessing) {
            updateFormElements(directory);
          }
        });
      });

      // Handle AJAX completion using once correctly
      once('directory-ajax', document).forEach(function (doc) {
        $(doc).on('ajaxComplete', function (event, xhr, settings) {
          if (settings.url && settings.url.includes('/media/add/') && currentDirectory) {
            setTimeout(() => updateFormElements(currentDirectory), 100);
          }

          // Reapply to modals that might have loaded
          if (settings.url && settings.url.includes('modal')) {
            setTimeout(() => {
              if (currentDirectory) {
                updateFormElements(currentDirectory);
              }
            }, 200);
          }
        });
      });

      // Ensure directory is set for modal dialogs
      once('directory-modal', '.ui-dialog-content', context).forEach(function (content) {
        if (currentDirectory) {
          setTimeout(() => updateFormElements(currentDirectory), 200);
        }
      });
    }
  };

})(jQuery, Drupal, once);
