<?php

namespace Drupal\Tests\advanced_file_destination\FunctionalJavascript;

use Drupal\Core\File\FileSystemInterface;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\file\Entity\File;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Tests the Advanced File Destination form functionality.
 *
 * @group advanced_file_destination
 */
class AdvancedFileDestinationFormTest extends WebDriverTestBase {
  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'node',
    'file',
    'field',
    'field_ui',
    'media',
    'media_library',
    'advanced_file_destination',
    'config',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    // Ignore schema errors during testing.
    $this->strictConfigSchema = FALSE;
    parent::setUp();

    // Create test content type.
    $this->drupalCreateContentType(['type' => 'article']);

    // Create admin user with required permissions.
    $adminUser = $this->drupalCreateUser([
      'access content',
      'access administration pages',
      'administer content types',
      'administer node fields',
      'create article content',
      'access media overview',
      'administer media',
      'access advanced file destination',
      'create advanced file destination directories',
    ]);
    $this->drupalLogin($adminUser);

    // Set up test directories.
    $test_dirs = ['public://test_uploads', 'public://test_dir'];
    foreach ($test_dirs as $dir) {
      $this->container->get('file_system')->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY);
    }

    // Add file field to article content type.
    $this->createFileField('field_test_file', 'node', 'article', [
      'file_directory' => 'test_uploads',
      'uri_scheme' => 'public',
    ]);
  }

  /**
   * Creates a file field.
   *
   * @param string $name
   *   The name of the field.
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle name.
   * @param array $settings
   *   Additional field settings.
   *
   * @return \Drupal\field\Entity\FieldConfig
   *   The field config entity.
   */
  protected function createFileField($name, $entity_type, $bundle, array $settings = []) {
    // Create file field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name' => $name,
      'entity_type' => $entity_type,
      'type' => 'file',
      'cardinality' => !empty($settings['cardinality']) ? $settings['cardinality'] : 1,
      'settings' => [],
    ]);
    $field_storage->save();

    $field_config = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $bundle,
      'label' => $name,
      'settings' => $settings,
    ]);
    $field_config->save();

    // Create the form display for the field.
    $this->container->get('entity_display.repository')
      ->getFormDisplay($entity_type, $bundle)
      ->setComponent($name, [
        'type' => 'file_generic',
        'settings' => [],
      ])
      ->save();

    // Create the display for the field.
    $this->container->get('entity_display.repository')
      ->getViewDisplay($entity_type, $bundle)
      ->setComponent($name, [
        'type' => 'file_default',
        'label' => 'hidden',
      ])
      ->save();

    return $field_config;
  }

  /**
   * Tests directory selection and file upload.
   */
  public function testDirectorySelectionAndFileUpload() {
    // Create test directory.
    $test_dir = 'public://test_uploads';
    $this->container->get('file_system')->prepareDirectory($test_dir, FileSystemInterface::CREATE_DIRECTORY);

    // Visit node creation page.
    $this->drupalGet('node/add/article');
    $assert_session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Wait for form to be ready.
    $this->getSession()->wait(2000);
    $field = $this->getSession()->getPage()->findField('Destination folder');
    // Check if field exists.
    if (empty($field)) {
      $assert_session->assert(FALSE, 'Destination folder field does not exist on the page');
    }

    // Create and attach test file.
    $filename = 'test.txt';
    $file = $this->createTestFile($filename);
    $page->attachFileToField('files[field_test_file_0]', \Drupal::service('file_system')->realpath($file->getFileUri()));

    // Select directory and wait for AJAX.
    $page->selectFieldOption('Destination folder', 'public://test_uploads');
    $this->getSession()->wait(5000, 'jQuery.active === 0');

    // Verify AJAX completed properly using JS condition.
    $this->assertJsCondition('jQuery.active === 0', 10000, 'AJAX request completed');

    // Fill required fields and submit.
    $page->fillField('Title', $this->randomString());
    $page->pressButton('Save');

    // Verify file location.
    $saved_file = File::load($file->id());
    $file_uri = $saved_file->getFileUri();
    $this->assertContains('test_uploads', $file_uri, 'File URI contains test_uploads directory');
  }

  /**
   * Creates a test file.
   */
  protected function createTestFile($filename = NULL) {
    $filename = $filename ?: $this->randomMachineName() . '.txt';
    $filepath = 'temporary://' . $filename;
    file_put_contents($filepath, $this->randomString());

    $file = File::create([
      'uri' => $filepath,
      'filename' => $filename,
      'status' => File::STATUS_PERMANENT,
    ]);
    $file->save();

    return $file;
  }

  /**
   * Tests creating new directory through modal.
   */
  public function testCreateNewDirectory() {
    $this->drupalGet('node/add/article');
    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    // Wait for form to be ready.
    $this->getSession()->wait(2000);
    $link = $this->getSession()->getPage()->find('css', 'a:contains("Create new folder")');
    if (empty($link)) {
      $assert_session->assert(FALSE, 'Create new folder link does not exist on the page');
    }

    // Click create new directory button and wait for modal.
    $page->clickLink('Create new folder');
    $this->getSession()->wait(2000);
    $dialog = $this->getSession()->getPage()->find('css', '.ui-dialog');
    if (empty($dialog)) {
      $assert_session->assert(FALSE, 'Dialog does not appear on page after clicking Create new folder');
    }

    // Verify modal is visible using JS condition.
    $this->assertJsCondition('jQuery(".ui-dialog").is(":visible")', 10000, 'Modal dialog is visible');

    // Fill in directory name and submit.
    $page->fillField('New directory name', 'new_test_dir');
    $page->pressButton('Create directory');
    $this->getSession()->wait(5000, 'jQuery.active === 0');

    // Verify AJAX completed properly using JS condition.
    $this->assertJsCondition('jQuery.active === 0', 10000, 'AJAX request completed');

    // Verify directory was created.
    $directory_exists = file_exists('public://new_test_dir');
    $assert_session->assert($directory_exists, 'Directory was created on filesystem');

    // Verify directory appears in select list.
    $option = $this->getSession()->getPage()->find('css', 'option[value="public://new_test_dir"]');
    if (empty($option)) {
      $assert_session->assert(FALSE, 'Directory option does not exist in select list');
    }
  }

  /**
   * Asserts that a string contains another string.
   *
   * @param string $needle
   *   The string to search for.
   * @param string $haystack
   *   The string to search in.
   * @param string $message
   *   The message to display if the assertion fails.
   */
  protected function assertContains($needle, $haystack, $message = '') {
    $this->assertSession()->assert(
      strpos($haystack, $needle) !== FALSE,
      $message ?: "String '$haystack' does not contain '$needle'."
    );
  }

}
