<?php

namespace Drupal\advanced_file_destination\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\EntityOwnerInterface;
use Drupal\user\UserInterface;

/**
 * Defines the Advanced File Destination Directory entity.
 *
 * @ContentEntityType(
 *   id = "afd_directory",
 *   label = @Translation("File Destination Directory"),
 *   handlers = {
 *     "storage" = "Drupal\advanced_file_destination\AFDDirectoryStorage",
 *     "access" = "Drupal\advanced_file_destination\Entity\Access\DirectoryAccessControlHandler",
 *     "list_builder" = "Drupal\advanced_file_destination\Controller\DirectoryListBuilder",
 *     "form" = {
 *       "add" = "Drupal\advanced_file_destination\Form\AdvancedFileDestinationDirectoryForm",
 *       "edit" = "Drupal\advanced_file_destination\Form\AdvancedFileDestinationDirectoryForm",
 *       "delete" = "Drupal\advanced_file_destination\Form\AdvancedFileDestinationDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider"
 *     },
 *     "views_data" = "Drupal\views\EntityViewsData"
 *   },
 *   base_table = "afd_directory",
 *   revision_table = "afd_directory_revision",
 *   admin_permission = "administer advanced file destination",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "label" = "name",
 *     "revision" = "revision_id",
 *     "status" = "status",
 *     "weight" = "weight",
 *     "owner" = "uid",
 *     "created" = "created",
 *     "changed" = "changed",
 *     "revision_translation_affected" = "revision_translation_affected",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log"
 *   },
 *   links = {
 *     "add-form" = "/admin/content/directories/add",
 *     "canonical" = "/admin/content/directories/{afd_directory}/view",
 *     "edit-form" = "/admin/content/directories/{afd_directory}/edit",
 *     "delete-form" = "/admin/content/directories/{afd_directory}/delete",
 *     "collection" = "/admin/content/directories",
 *     "revision" = "/admin/content/directories/{afd_directory}/revisions"
 *   },
 *   field_ui_base_route = "entity.afd_directory.collection"
 * )
 */
class AdvancedFileDestinationDirectory extends ContentEntityBase implements RevisionableInterface, EntityOwnerInterface {

  use EntityChangedTrait;
  use StringTranslationTrait;

  /**
   * Default value callback for the owner field.
   *
   * @return int
   *   The user ID of the current user.
   */
  public static function getDefaultEntityOwner() {
    return \Drupal::currentUser()->id();
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Add the revision_log field that is missing.
    $fields['revision_log'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Revision log message'))
      ->setDescription(t('The log message describing the changes in this revision.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('')
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 25,
        'settings' => [
          'rows' => 4,
        ],
      ]);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the directory.'))
      ->setRequired(TRUE)
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['path'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Path'))
      ->setDescription(t('The directory path.'))
      ->setRequired(TRUE)
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setDescription(t('Whether the directory is active.'))
      ->setDefaultValue(TRUE)
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
        'weight' => -3,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['weight'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Weight'))
      ->setDescription(t('The directory weight.'))
      ->setDefaultValue(0)
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -2,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['roles'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Roles'))
      ->setDescription(t('The roles that can access this directory.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user_role')
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setDisplayOptions('form', [
        'type' => 'options_buttons',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setDescription(t('The user ID of the directory author.'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(static::class . '::getDefaultEntityOwner')
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time the directory was created.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'timestamp',
        'weight' => -3,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Updated'))
      ->setDescription(t('The time the directory was last updated.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'timestamp',
        'weight' => -2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

  /**
   * Gets the directory path.
   *
   * @return string
   *   The directory path.
   */
  public function getPath() {
    return $this->get('path')->value;
  }

  /**
   * Sets the directory path.
   *
   * @param string $path
   *   The directory path.
   *
   * @return $this
   */
  public function setPath($path) {
    $this->set('path', $path);
    return $this;
  }

  /**
   * Gets the directory weight.
   *
   * @return int
   *   The directory weight.
   */
  public function getWeight() {
    return $this->get('weight')->value;
  }

  /**
   * Sets the directory weight.
   *
   * @param int $weight
   *   The directory weight.
   *
   * @return $this
   */
  public function setWeight($weight) {
    $this->set('weight', $weight);
    return $this;
  }

  /**
   * Returns whether the directory is active.
   *
   * @return bool
   *   TRUE if the directory is active, FALSE otherwise.
   */
  public function isActive() {
    return (bool) $this->get('status')->value;
  }

  /**
   * Sets the directory status.
   *
   * @param bool $status
   *   TRUE if the directory is active, FALSE otherwise.
   *
   * @return $this
   */
  public function setStatus($status) {
    $this->set('status', $status);
    return $this;
  }

  /**
   * Gets the roles that can access this directory.
   *
   * @return array
   *   The roles.
   */
  public function getRoles() {
    $roles = [];
    foreach ($this->get('roles') as $role) {
      $roles[] = $role->target_id;
    }
    return $roles;
  }

  /**
   * Sets the roles that can access this directory.
   *
   * @param array $roles
   *   The roles.
   *
   * @return $this
   */
  public function setRoles(array $roles) {
    $this->set('roles', $roles);
    return $this;
  }

  /**
   * Gets the revision log message.
   *
   * @return string
   *   The revision log message.
   */
  public function getRevisionLogMessage() {
    return $this->get('revision_log')->value;
  }

  /**
   * Sets the revision log message.
   *
   * @param string $revision_log
   *   The revision log message.
   *
   * @return $this
   */
  public function setRevisionLogMessage($revision_log) {
    $this->set('revision_log', $revision_log);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);

    // Clear cached directories using the service container directly.
    \Drupal::service('advanced_file_destination.manager')->clearCachedDirectories();
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->get('name')->value;
  }

  /**
   * Gets the name.
   *
   * @return string
   *   The name.
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * Sets the name.
   *
   * @param string $name
   *   The name.
   *
   * @return $this
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * Gets the creation timestamp.
   *
   * @return int
   *   The creation timestamp.
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * Gets the change timestamp.
   *
   * @return int
   *   The changed timestamp.
   */
  public function getChangedTime() {
    return $this->get('changed')->value;
  }

  /**
   * Gets the directory status.
   *
   * @return bool
   *   The directory status.
   */
  public function status() {
    return (bool) $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function isModeratable() {
    // This method tells Content Moderation we specifically support moderation.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    // When a new entity is created, set the owner ID to the current user.
    if ($this->isNew() && !$this->getOwnerId()) {
      $this->setOwnerId(\Drupal::currentUser()->id());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isNewRevision() {
    return $this->newRevision || (!$this->getRevisionId() && $this->isNew());
  }

  /**
   * {@inheritdoc}
   */
  public function setNewRevision($value = TRUE) {
    if ($this->isRevisionable()) {
      $this->set('revision', $value);
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('uid')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setRevisionUserId($uid) {
    if ($this->isRevisionable()) {
      $this->set('revision_uid', $uid);
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRevisionUser() {
    if ($this->isRevisionable() && $this->hasField('revision_uid')) {
      return $this->get('revision_uid')->entity;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getRevisionUserId() {
    if ($this->isRevisionable() && $this->hasField('revision_uid')) {
      return $this->get('revision_uid')->target_id;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isRevisionable() {
    return $this->getEntityType()->hasKey('revision');
  }

  /**
   * {@inheritdoc}
   */
  public function setRevisionCreationTime($timestamp) {
    if ($this->isRevisionable()) {
      $this->set('revision_timestamp', $timestamp);
    }
    return $this;
  }

}
