<?php

namespace Drupal\advanced_file_destination;

use Drupal\Core\Entity\ContentEntityStorageInterface;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\MemoryCache\MemoryCacheInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Storage handler for AFD Directory entities.
 */
class AFDDirectoryStorage extends SqlContentEntityStorage implements ContentEntityStorageInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a AFDDirectoryStorage object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection to be used.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend to be used.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Cache\MemoryCache\MemoryCacheInterface $memory_cache
   *   The memory cache.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface $entity_last_installed_schema_repository
   *   The entity last installed schema repository.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    Connection $database,
    EntityFieldManagerInterface $entity_field_manager,
    CacheBackendInterface $cache,
    LanguageManagerInterface $language_manager,
    MemoryCacheInterface $memory_cache,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    EntityTypeManagerInterface $entity_type_manager,
    EntityLastInstalledSchemaRepositoryInterface $entity_last_installed_schema_repository,
  ) {
    parent::__construct($entity_type, $database, $entity_field_manager, $cache, $language_manager, $memory_cache, $entity_type_bundle_info, $entity_type_manager, $entity_last_installed_schema_repository);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
          $entity_type,
          $container->get('database'),
          $container->get('entity_field.manager'),
          $container->get('cache.entity'),
          $container->get('language_manager'),
          $container->get('entity.memory_cache'),
          $container->get('entity_type.bundle.info'),
          $container->get('entity_type.manager'),
          $container->get('entity.last_installed_schema.repository')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getLatestRevisionId($entity_id) {
    if (!$entity_id) {
      return NULL;
    }

    $query = $this->database->select($this->entityType->getRevisionTable(), 'r')
      ->fields('r', [$this->entityType->getKey('revision')])
      ->condition('r.' . $this->entityType->getKey('id'), $entity_id)
      ->orderBy('r.' . $this->entityType->getKey('revision'), 'DESC')
      ->range(0, 1);

    return $query->execute()->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function loadMultipleRevisions(array $revision_ids) {
    if (empty($revision_ids)) {
      return [];
    }

    $revisions = [];
    $revision_ids_chunks = array_chunk($revision_ids, 50);

    foreach ($revision_ids_chunks as $chunk) {
      $entities = $this->loadMultiple($chunk);
      foreach ($entities as $entity) {
        $revisions[$entity->getRevisionId()] = $entity;
      }
    }

    return $revisions;
  }

  /**
   * {@inheritdoc}
   */
  public function loadAllRevisions($entity_id) {
    $entity_id = $entity_id ? $entity_id : FALSE;

    if (!$entity_id) {
      return [];
    }

    $table = $this->entityType->getRevisionTable();
    $id_field = $this->entityType->getKey('id');
    $revision_field = $this->entityType->getKey('revision');

    $query = $this->database->select($table, 'r')
      ->fields('r', [$revision_field])
      ->condition('r.' . $id_field, $entity_id)
      ->orderBy('r.' . $revision_field, 'DESC');

    $results = $query->execute()->fetchCol();
    return $results ? $this->loadMultipleRevisions($results) : [];
  }

}
