<?php

namespace Drupal\advanced_file_destination\Entity\Access;

use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;

/**
 * Access controller for the AFD Directory entity.
 */
class DirectoryAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    // Admin users have full access.
    if ($account->hasPermission('administer advanced file destination')) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    // For debugging purposes, temporarily allow all authenticated users access.
    if ($account->isAuthenticated()) {
      return AccessResult::allowed();
    }

    switch ($operation) {
      case 'view':
        return AccessResult::allowedIfHasPermission($account, 'access advanced file destination')
          ->orIf(AccessResult::allowedIfHasPermission($account, 'administer advanced file destination'))
          ->cachePerPermissions();

      case 'update':
        // Allow users who can create directories to also edit them.
        return AccessResult::allowedIfHasPermission($account, 'create advanced file destination directories')
          ->orIf(AccessResult::allowedIfHasPermission($account, 'administer advanced file destination'))
          ->cachePerPermissions();

      case 'delete':
        // Only admin users can delete directories.
        return AccessResult::allowedIfHasPermission($account, 'administer advanced file destination')
          ->cachePerPermissions();
    }

    return AccessResult::neutral();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    // For debugging purposes, temporarily allow all authenticated users access.
    if ($account->isAuthenticated()) {
      return AccessResult::allowed();
    }

    return AccessResult::allowedIfHasPermission($account, 'create advanced file destination directories')
      ->orIf(AccessResult::allowedIfHasPermission($account, 'administer advanced file destination'))
      ->cachePerPermissions();
  }

}
