<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for bulk updating directories.
 */
class DirectoryBulkUpdateForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new DirectoryBulkUpdateForm.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('entity_type.manager')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_bulk_update_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $selected_ids = $this->getRequest()->query->get('ids', []);
    if (empty($selected_ids)) {
      $this->messenger()->addError($this->t('No directories selected.'));
      return $form;
    }

    $form['ids'] = [
      '#type' => 'value',
      '#value' => $selected_ids,
    ];

    $form['action'] = [
      '#type' => 'select',
      '#title' => $this->t('Action'),
      '#options' => [
        'enable' => $this->t('Enable directories'),
        'disable' => $this->t('Disable directories'),
        'delete' => $this->t('Delete directories'),
      ],
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Apply'),
        '#button_type' => 'primary',
      ],
      'cancel' => [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => $this->getCancelUrl(),
        '#attributes' => ['class' => ['button']],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $ids = $form_state->getValue('ids');
    $action = $form_state->getValue('action');
    $storage = $this->entityTypeManager->getStorage('afd_directory');
    $directories = $storage->loadMultiple($ids);

    switch ($action) {
      case 'enable':
      case 'disable':
        foreach ($directories as $directory) {
          $directory->set('status', $action === 'enable');
          $directory->save();
        }
        $this->messenger()->addStatus(
              $this->t(
                  'Updated status for @count directories.', [
                    '@count' => count($directories),
                  ]
              )
          );
        break;

      case 'delete':
        $storage->delete($directories);
        $this->messenger()->addStatus(
              $this->t(
                  'Deleted @count directories.', [
                    '@count' => count($directories),
                  ]
              )
          );
        break;
    }

    $form_state->setRedirect('entity.afd_directory.collection');
  }

  /**
   * Gets the URL to redirect to when the form is cancelled.
   */
  protected function getCancelUrl() {
    return $this->getRequest()->query->get('destination') ?: 'entity.afd_directory.collection';
  }

}
