<?php

namespace Drupal\advanced_file_destination\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;

/**
 * Provides services for managing file destinations.
 */
class AdvancedFileDestinationManager {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The session.
   *
   * @var \Symfony\Component\HttpFoundation\Session\SessionInterface
   */
  protected $session;

  /**
   * The form ajax response builder.
   *
   * @var \Drupal\Core\Form\FormAjaxResponseBuilder
   */
  protected $formAjaxResponseBuilder;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The current directory being used.
   *
   * @var string|null
   */
  protected $currentDirectory = NULL;

  /**
   * Static cache of directories.
   *
   * @var array
   */
  protected static $directoriesCache = [];

  /**
   * Constructs a new AdvancedFileDestinationManager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\Session\SessionInterface $session
   *   The session.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The translation service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface|null $stream_wrapper_manager
   *   The stream wrapper manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    FileSystemInterface $file_system,
    AccountProxyInterface $current_user,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
    SessionInterface $session,
    EntityTypeManagerInterface $entity_type_manager,
    StateInterface $state,
    TranslationInterface $string_translation,
    ?StreamWrapperManagerInterface $stream_wrapper_manager = NULL,
  ) {
    $this->configFactory = $config_factory;
    $this->fileSystem = $file_system;
    $this->currentUser = $current_user;
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory->get('advanced_file_destination');
    $this->session = $session;
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
    $this->stringTranslation = $string_translation;
    $this->streamWrapperManager = $stream_wrapper_manager;
  }

  /**
   * Gets available directories for file uploads.
   *
   * @return array
   *   An array of directories keyed by path with name as value.
   */
  public function getAvailableDirectories() {
    // Check if we have cached directories.
    $cacheKey = $this->currentUser->id();
    if (isset(static::$directoriesCache[$cacheKey])) {
      return static::$directoriesCache[$cacheKey];
    }

    $config = $this->configFactory->get('advanced_file_destination.settings');
    $default_directory = $config->get('default_directory') ?: 'public://';
    $default_scheme = $this->getSchemeFromDirectory($default_directory);

    // Extract scheme name without :// for label display.
    $scheme_name = str_replace('://', '', $default_scheme);
    // Capitalize first letter for better presentation.
    $scheme_display = ucfirst($scheme_name);

    $root_dirs = [$default_directory => $scheme_display . ' files (default)'];

    if ($config->get('use_private') && $this->currentUser->hasPermission('access advanced file destination private files')) {
      $root_dirs['private://'] = 'Private files';
    }

    // Add configured directories.
    $directories = $root_dirs;

    // Get directories from database.
    $custom_dirs = $this->entityTypeManager
      ->getStorage('afd_directory')
      ->loadByProperties(['status' => 1]);

    foreach ($custom_dirs as $dir) {
      $directories[$dir->getPath()] = $dir->getName();
    }

    // Also scan filesystem for directories if enabled.
    if ($config->get('scan_filesystem')) {
      $scan_dirs = [];
      foreach (array_keys($root_dirs) as $root) {
        if (file_exists($root)) {
          $scanned = $this->fileSystem->scanDirectory($root, '/^.+$/', ['recurse' => FALSE, 'key' => 'uri']);
          foreach ($scanned as $uri => $file) {
            if (is_dir($uri)) {
              $name = basename($uri);
              $scan_dirs[$uri] = $name;
            }
          }
        }
      }

      $directories = array_merge($directories, $scan_dirs);
    }

    // Filter directories by permission.
    $directories = $this->filterDirectoriesByPermission($directories);

    // Cache the results.
    static::$directoriesCache[$cacheKey] = $directories;

    return $directories;
  }

  /**
   * Gets available directories including temporary ones for media items.
   *
   * @param string|null $additional_directory
   *   An additional directory to include in the list.
   *
   * @return array
   *   An array of directories keyed by path with name as value.
   */
  public function getAvailableDirectoriesWithTemporary($additional_directory = NULL) {
    $directories = $this->getAvailableDirectories();

    // Add additional directory if provided.
    if ($additional_directory) {
      // Normalize the directory path.
      $additional_directory = $this->normalizeDirectoryPath($additional_directory);

      // Check if directory exists physically.
      if (file_exists($additional_directory)) {
        // Get a human-readable name for the directory.
        $name = basename($additional_directory);
        // If this is a deep path, show the full path after the scheme.
        $scheme = $this->streamWrapperManager->getScheme($additional_directory);
        if ($scheme) {
          $relative_path = str_replace($scheme . '://', '', $additional_directory);
          if (strlen($relative_path) > strlen($name)) {
            $name = $relative_path;
          }
        }
        $directories[$additional_directory] = $name . ' (current location)';
      }
    }

    return $directories;
  }

  /**
   * Clears the cached directories.
   */
  public function clearCachedDirectories() {
    static::$directoriesCache = [];
  }

  /**
   * Filters directories based on user permissions.
   *
   * @param array $directories
   *   The directories to filter.
   *
   * @return array
   *   Filtered directories.
   */
  protected function filterDirectoriesByPermission(array $directories) {
    // If user has admin permission, return all directories.
    if ($this->currentUser->hasPermission('administer advanced file destination')) {
      return $directories;
    }

    // Get directory-specific permissions from configuration.
    $directory_permissions = $this->configFactory
      ->get('advanced_file_destination.settings')
      ->get('directory_permissions') ?: [];

    $filtered_directories = [];
    foreach ($directories as $path => $name) {
      // Private directory requires special permission.
      if (strpos($path, 'private://') === 0 && !$this->currentUser->hasPermission('access advanced file destination private files')) {
        continue;
      }

      // Check for directory-specific permission.
      $scheme = $this->getSchemeFromDirectory($path);
      $scheme_key = str_replace('://', '.', $scheme);
      $dir_key = str_replace([$scheme, 'private://'], [$scheme_key, 'private.'], $path);

      if (isset($directory_permissions[$dir_key])) {
        $required_permission = $directory_permissions[$dir_key];
        if (!empty($required_permission) && !$this->currentUser->hasPermission($required_permission)) {
          continue;
        }
      }

      $filtered_directories[$path] = $name;
    }

    return $filtered_directories;
  }

  /**
   * Gets the default directory for the current context.
   *
   * @return string
   *   The default directory path.
   */
  public function getDefaultDirectory() {
    $config = $this->configFactory->get('advanced_file_destination.settings');
    $default = $config->get('default_directory') ?: 'public://';

    // Check if the default directory is accessible to the current user.
    $available = $this->getAvailableDirectories();
    if (!isset($available[$default])) {
      // Fall back to the first available directory.
      return key($available);
    }

    return $default;
  }

  /**
   * Creates a directory.
   *
   * @param string $path
   *   The directory path to create.
   * @param string $name
   *   The human-readable name for the directory.
   *
   * @return bool
   *   TRUE if the directory was created successfully, FALSE otherwise.
   */
  public function createDirectory($path, $name = NULL) {
    if (!$this->currentUser->hasPermission('create advanced file destination directories')) {
      throw new \Exception('You do not have permission to create directories.');
    }

    // Ensure the path has a scheme.
    if (strpos($path, '://') === FALSE) {
      $default_directory = $this->getBaseDirectory();
      $default_scheme = $this->getSchemeFromDirectory($default_directory);
      $path = $default_scheme . $path;
    }

    // Check if private path and permissions.
    if (strpos($path, 'private://') === 0
          && !$this->currentUser->hasPermission('access advanced file destination private files')
      ) {
      throw new \Exception('You do not have permission to create private directories.');
    }

    // Create directory with proper permissions.
    if (!$this->fileSystem->prepareDirectory(
          $path,
          FileSystemInterface::CREATE_DIRECTORY |
          FileSystemInterface::MODIFY_PERMISSIONS
      )
      ) {
      throw new \Exception('Failed to create directory ' . $path);
    }

    // Save the directory entry.
    try {
      $directory = $this->entityTypeManager->getStorage('afd_directory')->create(
            [
              'name' => $name ?: basename($path),
              'path' => $path,
              'status' => 1,
              'created' => time(),
              'uid' => $this->currentUser->id(),
            ]
        );
      $directory->save();

      $this->messenger->addStatus(
            $this->stringTranslation->translate(
                'Directory %name created successfully.',
                ['%name' => $directory->label()]
            )
        );

      // Clear cached directories to force refresh.
      $this->clearCachedDirectories();

      return $path;
    }
    catch (\Exception $e) {
      throw new \Exception('Error saving directory: ' . $e->getMessage());
    }
  }

  /**
   * Gets the base upload directory from settings.
   *
   * @return string
   *   The base directory path.
   */
  public function getBaseDirectory() {
    $config = $this->configFactory->get('advanced_file_destination.settings');
    return $config->get('default_directory') ?: 'public://';
  }

  /**
   * Creates a new directory.
   *
   * @param string $name
   *   The name for the new directory.
   * @param string|null $parent_directory
   *   Optional parent directory where the new directory will be created.
   *
   * @return string
   *   The path of the newly created directory.
   *
   * @throws \Exception
   *   If directory creation fails.
   */
  public function createNewDirectory($name, $parent_directory = NULL) {
    $sanitized_name = preg_replace('/[^a-z0-9_-]/i', '_', $name);

    // Ensure parent directory has proper scheme.
    if ($parent_directory) {
      $parent_directory = $this->normalizeDirectoryPath($parent_directory);

      // Ensure parent directory exists and is writable.
      if (!$this->fileSystem->prepareDirectory($parent_directory, FileSystemInterface::CREATE_DIRECTORY)) {
        throw new \Exception($this->stringTranslation->translate('Parent directory is not accessible or writable.'));
      }
      $new_directory = rtrim($parent_directory, '/') . '/' . $sanitized_name;
    }
    else {
      $base_dir = $this->getBaseDirectory();
      $base_dir = rtrim($base_dir, '/');
      $new_directory = $base_dir . '/' . $sanitized_name;
    }

    $new_directory = $this->normalizeDirectoryPath($new_directory);

    // Create the directory.
    if (!$this->fileSystem->prepareDirectory(
          $new_directory,
          FileSystemInterface::CREATE_DIRECTORY |
          FileSystemInterface::MODIFY_PERMISSIONS
      )
      ) {
      throw new \Exception($this->stringTranslation->translate('Failed to create directory.'));
    }

    // Store the new directory path.
    $this->setDestinationDirectory($new_directory);

    // Clear cached directories to force refresh.
    $this->clearCachedDirectories();

    // Add the new directory to available directories immediately.
    static::$directoriesCache[$this->currentUser->id()][$new_directory] = basename($new_directory);

    return $new_directory;
  }

  /**
   * Normalizes a directory path.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getBaseDirectory();
    }

    // Get the default directory scheme.
    $default_directory = $this->getBaseDirectory();
    $default_scheme = $this->getSchemeFromDirectory($default_directory);

    // Remove any stacked scheme prefixes.
    $path = preg_replace('#^([a-z]+://)+#', '$1', $path);

    // If no scheme, add default scheme.
    if (!preg_match('#^[a-z]+://#', $path)) {
      $path = $default_scheme . ltrim($path, '/');
    }

    // Ensure single trailing slash.
    return rtrim($path, '/') . '/';
  }

  /**
   * Sets the destination directory.
   */
  public function setDestinationDirectory($directory) {
    if (empty($directory)) {
      $this->loggerFactory->error('Empty directory provided to setDestinationDirectory');
      return FALSE;
    }

    try {
      $default_directory = $this->getBaseDirectory();

      // Special case for public directory.
      if ($directory == 'public://' || $directory == 'public:///' || $directory == 'public://public:/') {
        $directory = 'public://';
      }

      // Don't reset to default directory if already have specific directory.
      if ($directory === $default_directory && $this->currentDirectory && $this->currentDirectory !== $default_directory) {
        $this->loggerFactory->debug(
              'Prevented reset to @default from @current', [
                '@default' => $default_directory,
                '@current' => $this->currentDirectory,
              ]
          );
        return TRUE;
      }

      // Normalize directory path with trailing slash.
      $directory = $this->normalizeDirectoryPath($directory);

      // Additional check for stacked schemes (public://public://)
      if (strpos($directory, 'public://public://') === 0) {
        $directory = 'public://';
      }

      // Don't process if directory hasn't changed.
      if ($this->currentDirectory === $directory) {
        return TRUE;
      }

      // Store in class property first.
      $this->currentDirectory = $directory;

      // Store in state - this is the authoritative source now.
      $this->state->set('advanced_file_destination.directory', $directory);
      $this->state->set('advanced_file_destination.directory_timestamp', time());

      // Set static upload defaults directly.
      $defaults = &drupal_static('file_upload_defaults', []);
      $defaults['upload_location'] = $directory;
      $defaults['file_directory'] = $directory;

      // Add to static directories cache.
      static::$directoriesCache[$this->currentUser->id()][$directory] = basename($directory);

      $this->loggerFactory->notice(
            'Directory explicitly set and enforced: @dir',
            ['@dir' => $directory]
        );

      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->error(
            'Failed to set directory: @error',
            ['@error' => $e->getMessage()]
        );
      return FALSE;
    }
  }

  /**
   * Gets the scheme from a directory path.
   *
   * @param string $path
   *   The directory path.
   *
   * @return string
   *   The scheme with :// suffix, or empty string if no scheme found.
   */
  protected function getSchemeFromDirectory($path) {
    if (preg_match('#^([a-z]+)://#', $path, $matches)) {
      return $matches[1] . '://';
    }
    return '';
  }

  /**
   * Gets the current directory without falling back to default.
   */
  public function getCurrentDirectory() {
    return $this->currentDirectory ?: $this->state->get('advanced_file_destination.directory');
  }

  /**
   * Gets the destination directory.
   */
  public function getDestinationDirectory($form_state = NULL) {
    // Check current instance first.
    if ($this->currentDirectory) {
      // Handle special cases for root directories.
      if ($this->currentDirectory == 'public:///' || $this->currentDirectory == 'public://public:/') {
        return 'public://';
      }
      return $this->currentDirectory;
    }

    // Check form state if provided.
    if ($form_state && $form_state->has(['advanced_file_destination', 'directory'])) {
      $directory = $form_state->get(['advanced_file_destination', 'directory']);
      if ($directory) {
        // Handle special cases for root directories.
        if ($directory == 'public:///' || $directory == 'public://public:/') {
          $directory = 'public://';
        }
        $this->currentDirectory = $directory;
        return $directory;
      }
    }

    // Check state - this is now the authoritative source.
    $directory = $this->state->get('advanced_file_destination.directory');
    if ($directory) {
      // Handle special cases for root directories.
      if ($directory == 'public:///' || $directory == 'public://public:/') {
        $directory = 'public://';
      }
      $this->currentDirectory = $directory;
      return $directory;
    }

    // Use default.
    $default = $this->getDefaultDirectory();
    $this->setDestinationDirectory($default);
    return $default;
  }

  /**
   * Clears the destination directory from the session.
   */
  public function clearDestinationDirectory() {
    $this->session->remove('advanced_file_destination.directory');
  }

  /**
   * Builds an AJAX response.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  protected function buildAjaxResponse($form, FormStateInterface $form_state) {
    $response = $this->formAjaxResponseBuilder->buildResponse($form, $form_state, []);
    return $response;
  }

  /**
   * Handles directory operations and returns an AJAX response.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function handleDirectoryOperation($form, FormStateInterface $form_state) {
    // Your existing logic here.
    return $this->buildAjaxResponse($form, $form_state);
  }

  /**
   * Ensures that a default directory is set.
   *
   * @return string
   *   The current or newly set default directory.
   */
  public function ensureDefaultDirectory() {
    $directory = $this->getDestinationDirectory();
    if (empty($directory)) {
      $directory = $this->getDefaultDirectory();
      $this->setDestinationDirectory($directory);
    }
    return $directory;
  }

  /**
   * Reset all directory storage.
   */
  public function clearCurrentDirectory() {
    $this->currentDirectory = NULL;
    $this->session->remove('advanced_file_destination.directory');
    $this->state->delete('advanced_file_destination.current_directory');

    $settings = &drupal_static('file_upload_defaults', []);
    unset($settings['upload_location']);
    unset($settings['file_directory']);
    $this->loggerFactory->notice('Cleared current directory selection');
  }

}
