<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Controller for directory revision overview.
 */
class DirectoryRevisionController extends ControllerBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a DirectoryRevisionController object.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    DateFormatterInterface $date_formatter,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
  ) {
    $this->dateFormatter = $date_formatter;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('date.formatter'),
          $container->get('entity_type.manager'),
          $container->get('current_user')
      );
  }

  /**
   * Generates an overview of all revisions of a directory.
   */
  public function revisionOverview(AdvancedFileDestinationDirectory $afd_directory) {
    $build['#theme'] = 'table';
    $build['#header'] = [
      $this->t('Status'),
      $this->t('Operations'),
      $this->t('Date'),
      $this->t('User'),
      $this->t('Log message'),
    ];

    $build['#rows'] = [];

    // Build username render array with fallback.
    $username = [
      '#theme' => 'username',
      '#account' => $afd_directory->getRevisionUser() ?? $this->currentUser,
    ];

    $row = [];

    // Status.
    $row[] = ['data' => ['#markup' => $this->t('Current version')]];

    // Operations.
    $links = [];
    $links['view'] = [
      'title' => $this->t('View'),
      'url' => $afd_directory->toUrl('edit-form'),
    ];
    $row[] = [
      'data' => [
        '#type' => 'operations',
        '#links' => $links,
      ],
    ];

    // Date.
    $changed_time = $afd_directory->getChangedTime() ?: $afd_directory->getCreatedTime();
    $row[] = $this->dateFormatter->format($changed_time, 'short');

    // User.
    $row[] = ['data' => $username];

    // Log message.
    $row[] = ['data' => ['#markup' => $afd_directory->getRevisionLogMessage() ?: '']];

    $build['#rows'][] = $row;

    // Add a note about config entities.
    $build['note'] = [
      '#type' => 'markup',
      '#markup' => $this->t('Note: Configuration entities store only their current version.'),
      '#prefix' => '<div class="revision-note">',
      '#suffix' => '</div>',
    ];

    return $build;
  }

}
