<?php

namespace Drupal\Tests\advanced_file_destination\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\Tests\field_ui\Traits\FieldUiTestTrait;
use Drupal\Tests\file\Functional\FileFieldCreationTrait;

/**
 * Tests the Advanced File Destination UI functionality.
 *
 * @group advanced_file_destination
 */
class AdvancedFileDestinationBrowserTest extends WebDriverTestBase {

  use TestFileCreationTrait;
  use FieldUiTestTrait;
  use FileFieldCreationTrait;

  /**
   * Modules to enable for this test.
   *
   * @var array
   */
  protected static $modules = [
    'node',
    'file',
    'field_ui',
    'media',
    'advanced_file_destination',
  ];

  /**
   * {@inheritdoc}
   */
  protected $strictConfigSchema = FALSE;

  /**
   * The theme to use for this test.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * Set up the test environment.
   *
   * Creates test with required permissions and content type with file field.
   */
  protected function setUp(): void {
    parent::setUp();

    // Create test content type.
    $this->drupalCreateContentType(['type' => 'article']);

    // Create test user.
    $admin_user = $this->drupalCreateUser(
          [
            'access content',
            'access administration pages',
            'administer site configuration',
            'administer content types',
            'administer node fields',
            'administer nodes',
          // This was causing the issue - needed to create content type first.
            'create article content',
            'access advanced file destination',
            'create advanced file destination directories',
          ]
      );
    $this->drupalLogin($admin_user);

    // Create file field.
    $this->createFileField('field_test_file', 'node', 'article');
  }

  /**
   * Tests the file upload process with custom directory selection.
   *
   * Verifies that files are properly uploaded to the selected destination
   * directory when using the Advanced File Destination module.
   */
  public function testFileUploadWithDirectorySelection() {
    $page = $this->getSession()->getPage();

    // Create test directories.
    $dir = 'public://';
    $this->container->get('file_system')->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY);

    // Navigate to node creation page.
    $this->drupalGet('node/add/article');

    // Select directory.
    $page->selectFieldOption('Destination folder', 'public://');
    $this->waitForAjax();

    // Upload file.
    $file = current($this->getTestFiles('text'));
    $page->attachFileToField('files[field_test_file_0]', \Drupal::service('file_system')->realpath($file->uri));
    $this->waitForAjax();

    // Verify file location by waiting for element to appear.
    // Wait for the file element to become visible.
    $page->waitFor(
          10, function () use ($page) {
              return $page->find('css', '.file--mime-text-plain');
          }
      );

    $uploaded_file = $page->find('css', '.js-form-managed-file a');

    if (empty($uploaded_file)) {
      throw new \Exception('File upload element was not found after AJAX completion');
    }

    $href = $uploaded_file->getAttribute('href');
    $this->assertNotEmpty($href, 'File upload element href is not empty');
    $this->assertStringContainsString('sites', $href, 'File upload element href contains public://');
  }

  /**
   * Helper method to wait for AJAX requests to complete.
   */
  protected function waitForAjax() {
    $condition = "(0 === jQuery.active)";
    $this->assertJsCondition($condition, 10000);
  }

}
