<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\advanced_file_destination\Form\NewDirectoryModalForm;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\OpenDialogCommand;
use Drupal\Core\Render\Markup;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Psr\Log\LoggerInterface;

/**
 * Controller for directory creation modal.
 */
class DirectoryModalController extends ControllerBase {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface;
   */
  protected $formBuilder;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The session service.
   *
   * @var \Symfony\Component\HttpFoundation\Session\SessionInterface
   */
  protected $session;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new DirectoryModalController.
   *
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Symfony\Component\HttpFoundation\Session\SessionInterface $session
   *   The session service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    FormBuilderInterface $form_builder,
    CsrfTokenGenerator $csrf_token,
    AdvancedFileDestinationManager $destination_manager,
    ConfigFactoryInterface $config_factory,
    RendererInterface $renderer,
    StateInterface $state,
    SessionInterface $session,
    LoggerInterface $logger,
  ) {
    $this->formBuilder = $form_builder;
    $this->csrfToken = $csrf_token;
    $this->destinationManager = $destination_manager;
    $this->configFactory = $config_factory;
    $this->renderer = $renderer;
    $this->state = $state;
    $this->session = $session;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('form_builder'),
          $container->get('csrf_token'),
          $container->get('advanced_file_destination.manager'),
          $container->get('config.factory'),
          $container->get('renderer'),
          $container->get('state'),
          $container->get('session'),
          $container->get('logger.factory')->get('advanced_file_destination')
      );
  }

  /**
   * Get the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Displays the modal form.
   */
  public function modal(Request $request, $adf_instance_id = NULL, $parent_directory = NULL) {

    if(!$parent_directory || empty($parent_directory) || !$adf_instance_id || empty($adf_instance_id)) {
      if(!$adf_instance_id || empty($adf_instance_id)) {
        // If no instance ID is provided, use the one from the request.
        $adf_instance_id = $request->get('adf_instance_id');
      }
      if(!$parent_directory || empty($parent_directory)) {
        // If no parent directory is provided, use the one from the request.
        $parent_directory = $request->get('parent_directory');  
      }    
      if(!$parent_directory || empty($parent_directory)) {
        // If no parent directory is provided, use the directory from the request.
        $parent_directory = \Drupal::request()->get('parent_directory');
      }
      if(!$adf_instance_id || empty($adf_instance_id)) {
        // If no instance ID is provided, use the one from the request.
        $adf_instance_id = \Drupal::request()->get('adf_instance_id');
      }
    }
    
    // Log debugging information
    $this->logger->debug('Parent directory: @parent_directory', ['@parent_directory' => $parent_directory ?? 'nada']);
    $this->logger->debug('ADF instance ID: @adf_instance_id', ['@adf_instance_id' => $adf_instance_id ?? 'nada']);
    
    
    $response = new AjaxResponse();
    
    // Build the form
    $modal_form = NewDirectoryModalForm::class;
    $form = $this->formBuilder->getForm($modal_form, $parent_directory, $adf_instance_id);
    if ($request->isMethod('POST') && $request->get('form_id') === 'new_directory_modal_form' && $request->get('_triggering_element_name') === 'save-directory') {
      // Handle form submission.
      return $response;
    }
    // In Drupal 11, è importante che le opzioni del dialog siano impostate correttamente
    $options = [
      'width' => 'auto',
      'closeOnEscape' => TRUE,      
      'title' => $this->t('Create a new directory'),
      'dialogClass' => 'advanced-file-destination-modal',
      'modal' => TRUE,
    ];

    // Usa OpenModalDialogCommand per garantire che sia trattato come un modale
    $response->addCommand(
      new OpenDialogCommand(
        '#new-directory-modal-form-wrapper',
        $this->t('Create a new directory'),
        $form,
        $options
      )
    );
    
    return $response;
  }
  
  /**
   * Helper function to normalize directory paths.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getDefaultDirectory();
    }

    // Remove multiple consecutive slashes.
    $path = preg_replace('#/+#', '/', $path);
    // Remove trailing slash.
    $path = rtrim($path, '/');

    // Get the default directory scheme.
    $default_dir = $this->getDefaultDirectory();
    $default_scheme = 'public://';

    // Extract scheme from default directory.
    if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
      $default_scheme = $matches[1] . '://';
    }

    // Handle scheme formatting.
    if (!preg_match('#^([a-z]+)://#', $path)) {
      $path = $default_scheme . ltrim($path, '/');
    }

    return $path;
  }
}
