<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Psr\Log\LoggerInterface;

/**
 * Controller for managing directory state.
 */
class DirectoryStateController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The advanced file destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new DirectoryStateController.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AdvancedFileDestinationManager $destination_manager,
    LoggerInterface $logger,
  ) {
    $this->configFactory = $config_factory;
    $this->destinationManager = $destination_manager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('advanced_file_destination.manager'),
          $container->get('logger.factory')->get('advanced_file_destination')
      );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Gets valid stream wrappers from configuration.
   *
   * @return array
   *   Array of valid stream wrapper schemes.
   */
  protected function getValidStreamWrappers() {
    $config = $this->configFactory->get('advanced_file_destination.settings');
    $wrappers = [];

    // Add all configured stream wrappers.
    if ($config->get('include_public')) {
      $wrappers[] = 'public';
    }
    if ($config->get('include_private')) {
      $wrappers[] = 'private';
    }
    if ($config->get('include_assets')) {
      $wrappers[] = 'assets';
    }
    if ($config->get('include_temporary')) {
      $wrappers[] = 'temporary';
    }

    // If no wrappers are configured, default to public and private.
    if (empty($wrappers)) {
      $wrappers = ['public', 'private'];
    }

    return $wrappers;
  }

  /**
   * Normalizes a directory path.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getDefaultDirectory();
    }

    // Get valid stream wrappers.
    $valid_wrappers = $this->getValidStreamWrappers();
    $valid_wrappers_regex = implode('|', $valid_wrappers);


    // Get the default directory scheme.
    $default_dir = $this->getDefaultDirectory();
    $default_scheme = 'public://';

    // Extract scheme from default directory.
    if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
      $default_scheme = $matches[1] . '://';
    }

    // Add scheme if missing.
    return $default_scheme . ltrim($path, '/');
  }

  /**
   * Updates directory selection state.
   */
  public function updateState(Request $request) {

    $adf_instance_id = $request->get('adf_instance_id');
    $directory = $request->get('directory');

    if(empty($adf_instance_id) || empty($directory)) {
      return new JsonResponse(
            [
              'status' => 'error',
              'message' => $this->t('Invalid form ID or directory.'),
              'error' => 'Invalid form ID or directory.',
            ], 400
        );
    }

    try {
      $this->destinationManager->setDestinationDirectory($directory, $adf_instance_id);

      return new JsonResponse(
           [ 'data' => [
              'status' => 'success',
              'directory' => $directory,
            ],
            'message' => $this->t('Directory updated successfully.'),
            'directory' => $directory,
            'adf_instance_id' => $adf_instance_id,
            'status' => 'success',
          ], 200
        );
    }
    catch (\Exception $e) {
      return new JsonResponse(
            [
              'status' => 'error',
              'message' => $e->getMessage(),
            ], 500
        );
    }
  }

}
