<?php

namespace Drupal\advanced_file_destination\EventSubscriber;

use Drupal\Core\File\FileExists;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for advanced file destination.
 */
class AdvancedFileDestinationSubscriber implements EventSubscriberInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new AdvancedFileDestinationSubscriber.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The advanced file destination manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    AccountProxyInterface $current_user,
    AdvancedFileDestinationManager $destination_manager,
    ConfigFactoryInterface $config_factory,
    FileSystemInterface $file_system,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->currentUser = $current_user;
    $this->destinationManager = $destination_manager;
    $this->configFactory = $config_factory;
    $this->fileSystem = $file_system;
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      // Higher priority to run first.
      'file.presave' => ['onFilePresave', 200],
    ];
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Reacts to the file presave event.
   *
   * @param object $event
   *   The file presave event.
   */
  public function onFilePresave(
    $event,
  ) {
    $file = $event->getFile();

    if (!$file->isNew()) {
      return;
    }

    try {
      // Get directory from state for the current user.
      $user_id = $this->currentUser->id();
          $current_form_request_id = \Drupal::request()->request->get('adf_instance_id')
    ?? \Drupal::request()->query->get('adf_instance_id');
      $selected_directory = $this->state->get('advanced_file_destination.directory.' . $user_id . '.' . $current_form_request_id);

      if (!$selected_directory) {
        $selected_directory = $this->destinationManager->getCurrentDirectory();
      }

      // Get default directory from settings.
      $default_directory = $this->getDefaultDirectory();

      if (!$selected_directory || $selected_directory === $default_directory) {
        return;
      }

      $current_uri = $file->getFileUri();
      if (!$current_uri) {
        return;
      }

      // Don't move if already in correct directory.
      if (strpos($current_uri, $selected_directory) === 0) {
        return;
      }

      $filename = basename($current_uri);
      $new_uri = rtrim($selected_directory, '/') . '/' . $filename;

      // Handle duplicates using FileSystemInterface.
      if (file_exists($new_uri)) {
        $new_uri = $this->fileSystem->getDestinationFilename($new_uri, FileExists::Rename);
      }

      // Move the file only if it exists.
      if (file_exists($current_uri)) {
        if (!$this->fileSystem->move($current_uri, $new_uri, FileExists::Rename)) {
          throw new \Exception('Failed to move file to ' . $new_uri);
        }
      }

      // Update the file entity.
      $file->setFileUri($new_uri);
      $file->setPermanent();

      $this->loggerFactory->get('advanced_file_destination')->notice(
            'File moved during presave: @new (from: @old)',
            ['@new' => $new_uri, '@old' => $current_uri]
        );
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('advanced_file_destination')->error(
            'File move failed: @error',
            ['@error' => $e->getMessage()]
        );
    }
  }

}
