<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Render\RendererInterface;

/**
 * Form for creating new directories.
 */
class NewDirectoryForm extends FormBase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The advanced file destination manager service.
   *
   * @var \Drupal\advanced_file_destination\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The private tempstore service.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new NewDirectoryForm.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\advanced_file_destination\AdvancedFileDestinationManager $destination_manager
   *   The advanced file destination manager service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    FileSystemInterface $file_system,
    AdvancedFileDestinationManager $destination_manager,
    PrivateTempStoreFactory $temp_store_factory,
    RendererInterface $renderer,
  ) {
    $this->fileSystem = $file_system;
    $this->destinationManager = $destination_manager;
    $this->tempStoreFactory = $temp_store_factory;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('file_system'),
          $container->get('advanced_file_destination.manager'),
          $container->get('tempstore.private'),
          $container->get('renderer')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_new_directory_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Directory name'),
      '#required' => TRUE,
      '#description' => $this->t('Enter the name for the new directory.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create directory'),
      '#ajax' => [
        'callback' => '::submitAjax',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Implementation handled in submitAjax.
  }

  /**
   * Ajax submit callback.
   */
  public function submitAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    $name = $form_state->getValue('name');
    $sanitized_name = preg_replace('/[^a-z0-9_-]/i', '_', $name);
    $date_dir = date('Y-m');

    $base_dir = $this->destinationManager->getBaseDirectory();
    $new_directory = $base_dir . '/' . $date_dir . '/' . $sanitized_name;

    try {
      if (!$this->fileSystem->prepareDirectory($new_directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        throw new \Exception('Failed to create directory.');
      }

      // Store the new directory in tempstore.
      $this->tempStoreFactory->get('advanced_file_destination')
        ->set('selected_directory', $new_directory);

      $response->addCommand(new CloseModalDialogCommand());
      $response->addCommand(new MessageCommand($this->t('Directory created successfully.'), NULL, ['type' => 'status']));

      // Update directory path display.
      $response->addCommand(
            new ReplaceCommand(
                '.directory-path-display',
                '<div class="directory-path-display visible">Created directory: ' . $new_directory . '</div>'
            )
        );

      // Update the directory selection dropdown.
      $directories = $this->destinationManager->getAvailableDirectories();
      $element = [
        '#type' => 'select',
        '#options' => $directories,
        '#value' => $new_directory,
      ];
      $response->addCommand(
            new ReplaceCommand(
                'select[name="advanced_file_destination[directory]"]',
                $this->renderer->render($element)
            )
        );
    }
    catch (\Exception $e) {
      $response->addCommand(new MessageCommand($e->getMessage(), NULL, ['type' => 'error']));
    }

    return $response;
  }

}
