<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\File\FileExists;
use Drupal\Core\Ajax\MessageCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Drupal\Core\State\StateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Form for creating a new directory in a modal.
 */
class NewDirectoryModalForm extends FormBase {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The private tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $afdManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $timeService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The session manager.
   *
   * @var \Symfony\Component\HttpFoundation\Session\Session
   */
  protected $sessionManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new NewDirectoryModalForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $afd_manager
   *   The advanced file destination manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system.
   * @param \Drupal\Component\Datetime\TimeInterface $time_service
   *   The time service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Symfony\Component\HttpFoundation\Session\Session $session_manager
   *   The session manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RequestStack $request_stack,
    PrivateTempStoreFactory $temp_store_factory,
    AdvancedFileDestinationManager $afd_manager,
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system,
    TimeInterface $time_service,
    EntityTypeManagerInterface $entity_type_manager,
    StateInterface $state,
    Session $session_manager,
    RendererInterface $renderer,
    AccountProxyInterface $current_user,
    MessengerInterface $messenger
  ) {
    $this->configFactory = $config_factory;
    $this->requestStack = $request_stack;
    $this->tempStoreFactory = $temp_store_factory;
    $this->afdManager = $afd_manager;
    $this->loggerFactory = $logger_factory;
    $this->fileSystem = $file_system;
    $this->timeService = $time_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
    $this->sessionManager = $session_manager;
    $this->renderer = $renderer;
    $this->currentUser = $current_user;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('request_stack'),
          $container->get('tempstore.private'),
          $container->get('advanced_file_destination.manager'),
          $container->get('logger.factory'),
          $container->get('file_system'),
          $container->get('datetime.time'),
          $container->get('entity_type.manager'),
          $container->get('state'),
          $container->get('session'),
          $container->get('renderer'),
          $container->get('current_user'),
          $container->get('messenger')
      );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'new-directory-modal-form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {  
    $parent_directory = \Drupal::request()->get('parent_directory');
    $adf_instance_id = \Drupal::request()->get('adf_instance_id');

    if (!$parent_directory || empty($parent_directory)) {
      $parent_directory = $this->state->get('advanced_file_destination.directory.' . $this->currentUser->id() . '.' . $adf_instance_id);
    }
    if (!$parent_directory) {
      $parent_directory = $this->afdManager->getDestinationDirectory($adf_instance_id);
    }
    if($parent_directory) {
      $form_state->setValue('parent_directory', $parent_directory);
    }
    if($adf_instance_id) {
      $form_state->setValue('adf_instance_id', $adf_instance_id);
    }
    
    $form['#id'] = $this->getFormId();
    
    // Set the form ID and wrapping div
    $form['#prefix'] = '<div class="advanced-file-destination-modal">'; 
    $form['#suffix'] = '</div>';

    $form['parent_directory'] = [
      '#type' => 'hidden',
      '#id' => 'parent-directory',
      '#name' => 'parent_directory',
      '#value' => $parent_directory,
    ];

    $form['parent_path'] = [
      '#type' => 'item',
      '#title' => $this->t('Parent directory'),
      '#description' => $this->t('The directory where the new directory will be created.'),
      '#title_display' => 'before',
      '#id' => 'parent-directory-display',
      '#name' => 'parent_directory_display',
      '#markup' => '<code>' . $parent_directory . '</code>',
      '#attributes' => ['class' => ['parent-directory-display']],
    ];

    $form['directory_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('New directory name'),
      '#description' => $this->t('Enter the name of the new directory.'),
      '#required' => TRUE,
      '#id' => 'directory-name',
      '#name' => 'directory_name',
      '#title_display' => 'before',
      '#maxlength' => 255,
      '#attributes' => [
        'class' => ['js-directory-name'],
        'placeholder' => $this->t('e.g., project-2024'),
      ],
    ];

    $form['preview'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['directory-preview']],
      'label' => [
        '#markup' => '<strong>' . $this->t('Full path preview:') . '</strong>',
      ],
      'path' => [
        '#prefix' => '<div class="path-preview">',
        '#markup' => $parent_directory,
        '#suffix' => '</div>',
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
      '#attributes' => ['class' => ['dialog-actions', 'form-actions']],
      '#id' => 'edit-actions-modal',
    ];
    
    $form['actions']['save'] = [
      '#type' => 'submit',
      '#name' => 'save-directory',
      '#id' => 'save-directory-custom',
      '#value' => $this->t('Create dir'),
      '#attributes' => [
        'class' => ['button--primary'],
        'data-drupal-selector' => 'edit-save-directory',
      ],
      '#ajax' => [
        'callback' => '::submitDirectoryFormAjax',
        'wrapper' => 'new-directory-modal-form-wrapper',
        'method' => 'replace',
        'effect' => 'fade',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Creating directory...'),
        ],
      ],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'submit',
      '#value' => $this->t('Cancel'),
      '#attributes' => [
        'class' => ['dialog-cancel', 'button--secondary'],
        'id' => 'edit-cancel',
      ],
      '#submit' => ['::cancelForm'],
    ];

    // These settings are crucial for preventing redirect
    $form['#attached']['library'][] = 'core/drupal.dialog';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    $form['#attached']['library'][] = 'advanced_file_destination/directory_preview';
    
    // Per il debug, aggiungiamo un identificatore JavaScript
    $form['#attached']['drupalSettings']['afd_debug'] = TRUE;

    
    // Disable redirect in Drupal 11
    //$form_state->disableRedirect();
    //$form['#redirect'] = FALSE;
    
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $directory_name = $form_state->getValue('directory_name');
    $parent_directory = $form_state->getValue('parent_directory');

    // Validate the directory name.
    if (empty($directory_name)) {
      $form_state->setErrorByName('directory_name', $this->t('Directory name cannot be empty.'));
    }
    elseif (preg_match('/[^a-zA-Z0-9_\-]/', $directory_name)) {
      $form_state->setErrorByName('directory_name', $this->t('Directory name can only contain letters, numbers, underscores, and dashes.'));
    }

    // Validate the parent directory.
    if (empty($parent_directory)) {
      $form_state->setErrorByName('parent_directory', $this->t('Parent directory cannot be empty.'));
    }
  }

  /**
   * Normalizes a directory path.
   */
  protected function normalizeDirectoryPath($path) {
    if (empty($path)) {
      return $this->getDefaultDirectory();
    }

    if ($path === 'public:/') {
      return 'public://';
    }


    // Get the default directory scheme.
    $default_dir = $this->getDefaultDirectory();
    $default_scheme = 'public://';

    // Extract scheme from default directory.
    if (preg_match('#^([a-z]+)://#', $default_dir, $matches)) {
      $default_scheme = $matches[1] . '://';
    }
    // Special case handling for public directories.
    if ($path == 'public://' || $path == 'public:///'
          || $path == 'public://public:/' || $path == 'public://public://'
      ) {
      $path = 'public://';
      return $path;
    }

    // Add scheme if missing.
    return $default_scheme . ltrim($path, '/');
  }

  /**
   * Ajax submit callback.
   */
  public function submitDirectoryFormAjax(array &$form, FormStateInterface $form_state) {
    // Forza la disabilitazione di qualsiasi reindirizzamento
    //$form_state->disableRedirect();
    
    $parent_directory = $form_state->getValue('parent_directory');
    $directory_name = $form_state->getValue('directory_name');
    
    if (empty($directory_name)) {
      // Se il nome della directory è vuoto, ritorna il form con un messaggio di errore
      $form_state->setErrorByName('directory_name', $this->t('Directory name cannot be empty.'));
      return $form;
    }

    $directory_name = preg_replace('/[^a-zA-Z0-9_\-]/', '', $directory_name);
    

    $full_path = $parent_directory . $directory_name;
    
    // Create the directory
    try {
      $created = $this->fileSystem->prepareDirectory($full_path, FileSystemInterface::CREATE_DIRECTORY || FileSystemInterface::MODIFY_PERMISSIONS || FileExists::Replace);
      if ($created) {
        
        $response = new AjaxResponse(); 

        // Add the directory to the select options  
        $response->addCommand(new InvokeCommand(
          'select.js-advanced-file-destination-select',
          'append',
          [['<option value='.$full_path.' selected="selected">'.ucfirst($directory_name).'</option>']]
        ));
        // Update the preview with the new directory path
        $response->addCommand(new InvokeCommand(
          'select.js-advanced-file-destination-select',
          'trigger',
          ['change']
        ));

        $response->addCommand(new CloseDialogCommand('#new-directory-modal-form-wrapper'));

        $response->addCommand(new MessageCommand($this->t('Directory created successfully: @path', ['@path' => $full_path])));
          
        return $response;
      }
      else {
        $this->messenger->addError($this->t('Failed to create directory @path.', ['@path' => $full_path]));
        return $form;
      }
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Error creating directory: @error', ['@error' => $e->getMessage()]));
      $this->loggerFactory->get('advanced_file_destination')->error('Error creating directory: @error', ['@error' => $e->getMessage()]);
      return $form;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Prevent redirect
    //$form_state->disableRedirect();
  }

  /**
   * Cancel form callback.
   */
  public function cancelForm(array &$form, FormStateInterface $form_state) {
    $form_state->disableRedirect();
    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand('.advanced-file-destination-modal'));
    return $response;
  }

  /**
   * AJAX callback to close the modal.
   */
  public function closeDirectoryModal(array &$form, FormStateInterface $form_state) {

    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand('#afd-directory-dialog'));
    $form_state->setResponse($response);
    $form_state->setRebuild(TRUE);
    return $response;
  }

  /**
   * Builds the HTML for select options.
   *
   * @param array $directories
   *   Array of directories.
   * @param string $selected_value
   *   The selected directory value.
   *
   * @return string
   *   HTML string of option elements.
   */
  protected function buildOptionsHtml($directories, $selected_value) {
    $options = [];
    foreach ($directories as $value => $label) {
      $selected = $value === $selected_value ? ' selected="selected"' : '';
      $options[] = sprintf(
            '<option value="%s"%s>%s</option>',
            htmlspecialchars($value),
            $selected,
            htmlspecialchars($label)
        );
    }
    return implode("\n", $options);
  }

  /**
   * {@inheritdoc}
   */
  protected function normalizeWrapper($path) {
    $default_dir = $this->getDefaultDirectory();
    if ($path) {  
      // Cehck if path contains private:// string
      if (strpos($path, 'private://') !== FALSE) {
        $path = str_replace('private://', $default_dir, $path);
      }
      // Check if path contains temporary:// string
      elseif (strpos($path, 'temporary://') !== FALSE) {
        $path = str_replace('temporary://', $default_dir, $path);
      }
      // Check if path contains asset:// string
      elseif (strpos($path, 'asset://') !== FALSE) {
        $path = str_replace('asset://', $default_dir, $path);
      }
      return $path;
    } else {
      return $default_dir;
    }
  }
}
